// Copyright 2024 The Go Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package ssagen

import (
	"flag"
	"fmt"
	"slices"
	"strings"
	"testing"

	"cmd/internal/sys"
)

var updateIntrinsics = flag.Bool("update", false, "Print an updated intrinsics table")

type testIntrinsicKey struct {
	archName string
	pkg      string
	fn       string
}

var wantIntrinsics = map[testIntrinsicKey]struct{}{
	{"386", "internal/runtime/math", "MulUintptr"}:                     struct{}{},
	{"386", "internal/runtime/sys", "Bswap32"}:                         struct{}{},
	{"386", "internal/runtime/sys", "Bswap64"}:                         struct{}{},
	{"386", "internal/runtime/sys", "GetCallerPC"}:                     struct{}{},
	{"386", "internal/runtime/sys", "GetCallerSP"}:                     struct{}{},
	{"386", "internal/runtime/sys", "GetClosurePtr"}:                   struct{}{},
	{"386", "internal/runtime/sys", "TrailingZeros32"}:                 struct{}{},
	{"386", "internal/runtime/sys", "TrailingZeros64"}:                 struct{}{},
	{"386", "internal/runtime/sys", "TrailingZeros8"}:                  struct{}{},
	{"386", "math", "sqrt"}:                                            struct{}{},
	{"386", "math/bits", "ReverseBytes32"}:                             struct{}{},
	{"386", "math/bits", "ReverseBytes64"}:                             struct{}{},
	{"386", "math/bits", "TrailingZeros16"}:                            struct{}{},
	{"386", "math/bits", "TrailingZeros32"}:                            struct{}{},
	{"386", "math/bits", "TrailingZeros64"}:                            struct{}{},
	{"386", "math/bits", "TrailingZeros8"}:                             struct{}{},
	{"386", "runtime", "KeepAlive"}:                                    struct{}{},
	{"386", "runtime", "slicebytetostringtmp"}:                         struct{}{},
	{"amd64", "internal/runtime/atomic", "And"}:                        struct{}{},
	{"amd64", "internal/runtime/atomic", "And32"}:                      struct{}{},
	{"amd64", "internal/runtime/atomic", "And64"}:                      struct{}{},
	{"amd64", "internal/runtime/atomic", "And8"}:                       struct{}{},
	{"amd64", "internal/runtime/atomic", "Cas"}:                        struct{}{},
	{"amd64", "internal/runtime/atomic", "Cas64"}:                      struct{}{},
	{"amd64", "internal/runtime/atomic", "CasRel"}:                     struct{}{},
	{"amd64", "internal/runtime/atomic", "Casint32"}:                   struct{}{},
	{"amd64", "internal/runtime/atomic", "Casint64"}:                   struct{}{},
	{"amd64", "internal/runtime/atomic", "Casp1"}:                      struct{}{},
	{"amd64", "internal/runtime/atomic", "Casuintptr"}:                 struct{}{},
	{"amd64", "internal/runtime/atomic", "Load"}:                       struct{}{},
	{"amd64", "internal/runtime/atomic", "Load64"}:                     struct{}{},
	{"amd64", "internal/runtime/atomic", "Load8"}:                      struct{}{},
	{"amd64", "internal/runtime/atomic", "LoadAcq"}:                    struct{}{},
	{"amd64", "internal/runtime/atomic", "LoadAcq64"}:                  struct{}{},
	{"amd64", "internal/runtime/atomic", "LoadAcquintptr"}:             struct{}{},
	{"amd64", "internal/runtime/atomic", "Loadint32"}:                  struct{}{},
	{"amd64", "internal/runtime/atomic", "Loadint64"}:                  struct{}{},
	{"amd64", "internal/runtime/atomic", "Loadp"}:                      struct{}{},
	{"amd64", "internal/runtime/atomic", "Loaduint"}:                   struct{}{},
	{"amd64", "internal/runtime/atomic", "Loaduintptr"}:                struct{}{},
	{"amd64", "internal/runtime/atomic", "Or"}:                         struct{}{},
	{"amd64", "internal/runtime/atomic", "Or32"}:                       struct{}{},
	{"amd64", "internal/runtime/atomic", "Or64"}:                       struct{}{},
	{"amd64", "internal/runtime/atomic", "Or8"}:                        struct{}{},
	{"amd64", "internal/runtime/atomic", "Store"}:                      struct{}{},
	{"amd64", "internal/runtime/atomic", "Store64"}:                    struct{}{},
	{"amd64", "internal/runtime/atomic", "Store8"}:                     struct{}{},
	{"amd64", "internal/runtime/atomic", "StoreRel"}:                   struct{}{},
	{"amd64", "internal/runtime/atomic", "StoreRel64"}:                 struct{}{},
	{"amd64", "internal/runtime/atomic", "StoreReluintptr"}:            struct{}{},
	{"amd64", "internal/runtime/atomic", "Storeint32"}:                 struct{}{},
	{"amd64", "internal/runtime/atomic", "Storeint64"}:                 struct{}{},
	{"amd64", "internal/runtime/atomic", "StorepNoWB"}:                 struct{}{},
	{"amd64", "internal/runtime/atomic", "Storeuintptr"}:               struct{}{},
	{"amd64", "internal/runtime/atomic", "Xadd"}:                       struct{}{},
	{"amd64", "internal/runtime/atomic", "Xadd64"}:                     struct{}{},
	{"amd64", "internal/runtime/atomic", "Xaddint32"}:                  struct{}{},
	{"amd64", "internal/runtime/atomic", "Xaddint64"}:                  struct{}{},
	{"amd64", "internal/runtime/atomic", "Xadduintptr"}:                struct{}{},
	{"amd64", "internal/runtime/atomic", "Xchg"}:                       struct{}{},
	{"amd64", "internal/runtime/atomic", "Xchg64"}:                     struct{}{},
	{"amd64", "internal/runtime/atomic", "Xchg8"}:                      struct{}{},
	{"amd64", "internal/runtime/atomic", "Xchgint32"}:                  struct{}{},
	{"amd64", "internal/runtime/atomic", "Xchgint64"}:                  struct{}{},
	{"amd64", "internal/runtime/atomic", "Xchguintptr"}:                struct{}{},
	{"amd64", "internal/runtime/maps", "bitsetFirst"}:                  struct{}{},
	{"amd64", "internal/runtime/maps", "bitsetRemoveBelow"}:            struct{}{},
	{"amd64", "internal/runtime/maps", "bitsetLowestSet"}:              struct{}{},
	{"amd64", "internal/runtime/maps", "bitsetShiftOutLowest"}:         struct{}{},
	{"amd64", "internal/runtime/maps", "ctrlGroupMatchH2"}:             struct{}{},
	{"amd64", "internal/runtime/maps", "ctrlGroupMatchEmpty"}:          struct{}{},
	{"amd64", "internal/runtime/maps", "ctrlGroupMatchEmptyOrDeleted"}: struct{}{},
	{"amd64", "internal/runtime/maps", "ctrlGroupMatchFull"}:           struct{}{},
	{"amd64", "internal/runtime/math", "Add64"}:                        struct{}{},
	{"amd64", "internal/runtime/math", "Mul64"}:                        struct{}{},
	{"amd64", "internal/runtime/math", "MulUintptr"}:                   struct{}{},
	{"amd64", "internal/runtime/sys", "Bswap32"}:                       struct{}{},
	{"amd64", "internal/runtime/sys", "Bswap64"}:                       struct{}{},
	{"amd64", "internal/runtime/sys", "GetCallerPC"}:                   struct{}{},
	{"amd64", "internal/runtime/sys", "GetCallerSP"}:                   struct{}{},
	{"amd64", "internal/runtime/sys", "GetClosurePtr"}:                 struct{}{},
	{"amd64", "internal/runtime/sys", "Len64"}:                         struct{}{},
	{"amd64", "internal/runtime/sys", "Len8"}:                          struct{}{},
	{"amd64", "internal/runtime/sys", "OnesCount64"}:                   struct{}{},
	{"amd64", "internal/runtime/sys", "Prefetch"}:                      struct{}{},
	{"amd64", "internal/runtime/sys", "PrefetchStreamed"}:              struct{}{},
	{"amd64", "internal/runtime/sys", "TrailingZeros32"}:               struct{}{},
	{"amd64", "internal/runtime/sys", "TrailingZeros64"}:               struct{}{},
	{"amd64", "internal/runtime/sys", "TrailingZeros8"}:                struct{}{},
	{"amd64", "math", "Ceil"}:                                          struct{}{},
	{"amd64", "math", "FMA"}:                                           struct{}{},
	{"amd64", "math", "Floor"}:                                         struct{}{},
	{"amd64", "math", "RoundToEven"}:                                   struct{}{},
	{"amd64", "math", "Trunc"}:                                         struct{}{},
	{"amd64", "math", "sqrt"}:                                          struct{}{},
	{"amd64", "math/big", "mulWW"}:                                     struct{}{},
	{"amd64", "math/bits", "Add"}:                                      struct{}{},
	{"amd64", "math/bits", "Add64"}:                                    struct{}{},
	{"amd64", "math/bits", "Div"}:                                      struct{}{},
	{"amd64", "math/bits", "Div64"}:                                    struct{}{},
	{"amd64", "math/bits", "Len"}:                                      struct{}{},
	{"amd64", "math/bits", "Len16"}:                                    struct{}{},
	{"amd64", "math/bits", "Len32"}:                                    struct{}{},
	{"amd64", "math/bits", "Len64"}:                                    struct{}{},
	{"amd64", "math/bits", "Len8"}:                                     struct{}{},
	{"amd64", "math/bits", "Mul"}:                                      struct{}{},
	{"amd64", "math/bits", "Mul64"}:                                    struct{}{},
	{"amd64", "math/bits", "OnesCount"}:                                struct{}{},
	{"amd64", "math/bits", "OnesCount16"}:                              struct{}{},
	{"amd64", "math/bits", "OnesCount32"}:                              struct{}{},
	{"amd64", "math/bits", "OnesCount64"}:                              struct{}{},
	{"amd64", "math/bits", "ReverseBytes32"}:                           struct{}{},
	{"amd64", "math/bits", "ReverseBytes64"}:                           struct{}{},
	{"amd64", "math/bits", "RotateLeft"}:                               struct{}{},
	{"amd64", "math/bits", "RotateLeft16"}:                             struct{}{},
	{"amd64", "math/bits", "RotateLeft32"}:                             struct{}{},
	{"amd64", "math/bits", "RotateLeft64"}:                             struct{}{},
	{"amd64", "math/bits", "RotateLeft8"}:                              struct{}{},
	{"amd64", "math/bits", "Sub"}:                                      struct{}{},
	{"amd64", "math/bits", "Sub64"}:                                    struct{}{},
	{"amd64", "math/bits", "TrailingZeros16"}:                          struct{}{},
	{"amd64", "math/bits", "TrailingZeros32"}:                          struct{}{},
	{"amd64", "math/bits", "TrailingZeros64"}:                          struct{}{},
	{"amd64", "math/bits", "TrailingZeros8"}:                           struct{}{},
	{"amd64", "runtime", "KeepAlive"}:                                  struct{}{},
	{"amd64", "runtime", "slicebytetostringtmp"}:                       struct{}{},
	{"amd64", "sync", "runtime_LoadAcquintptr"}:                        struct{}{},
	{"amd64", "sync", "runtime_StoreReluintptr"}:                       struct{}{},
	{"amd64", "sync/atomic", "AddInt32"}:                               struct{}{},
	{"amd64", "sync/atomic", "AddInt64"}:                               struct{}{},
	{"amd64", "sync/atomic", "AddUint32"}:                              struct{}{},
	{"amd64", "sync/atomic", "AddUint64"}:                              struct{}{},
	{"amd64", "sync/atomic", "AddUintptr"}:                             struct{}{},
	{"amd64", "sync/atomic", "AndInt32"}:                               struct{}{},
	{"amd64", "sync/atomic", "AndInt64"}:                               struct{}{},
	{"amd64", "sync/atomic", "AndUint32"}:                              struct{}{},
	{"amd64", "sync/atomic", "AndUint64"}:                              struct{}{},
	{"amd64", "sync/atomic", "AndUintptr"}:                             struct{}{},
	{"amd64", "sync/atomic", "CompareAndSwapInt32"}:                    struct{}{},
	{"amd64", "sync/atomic", "CompareAndSwapInt64"}:                    struct{}{},
	{"amd64", "sync/atomic", "CompareAndSwapUint32"}:                   struct{}{},
	{"amd64", "sync/atomic", "CompareAndSwapUint64"}:                   struct{}{},
	{"amd64", "sync/atomic", "CompareAndSwapUintptr"}:                  struct{}{},
	{"amd64", "sync/atomic", "LoadInt32"}:                              struct{}{},
	{"amd64", "sync/atomic", "LoadInt64"}:                              struct{}{},
	{"amd64", "sync/atomic", "LoadPointer"}:                            struct{}{},
	{"amd64", "sync/atomic", "LoadUint32"}:                             struct{}{},
	{"amd64", "sync/atomic", "LoadUint64"}:                             struct{}{},
	{"amd64", "sync/atomic", "LoadUintptr"}:                            struct{}{},
	{"amd64", "sync/atomic", "OrInt32"}:                                struct{}{},
	{"amd64", "sync/atomic", "OrInt64"}:                                struct{}{},
	{"amd64", "sync/atomic", "OrUint32"}:                               struct{}{},
	{"amd64", "sync/atomic", "OrUint64"}:                               struct{}{},
	{"amd64", "sync/atomic", "OrUintptr"}:                              struct{}{},
	{"amd64", "sync/atomic", "StoreInt32"}:                             struct{}{},
	{"amd64", "sync/atomic", "StoreInt64"}:                             struct{}{},
	{"amd64", "sync/atomic", "StoreUint32"}:                            struct{}{},
	{"amd64", "sync/atomic", "StoreUint64"}:                            struct{}{},
	{"amd64", "sync/atomic", "StoreUintptr"}:                           struct{}{},
	{"amd64", "sync/atomic", "SwapInt32"}:                              struct{}{},
	{"amd64", "sync/atomic", "SwapInt64"}:                              struct{}{},
	{"amd64", "sync/atomic", "SwapUint32"}:                             struct{}{},
	{"amd64", "sync/atomic", "SwapUint64"}:                             struct{}{},
	{"amd64", "sync/atomic", "SwapUintptr"}:                            struct{}{},
	{"arm", "internal/runtime/sys", "Bswap32"}:                         struct{}{},
	{"arm", "internal/runtime/sys", "Bswap64"}:                         struct{}{},
	{"arm", "internal/runtime/sys", "GetCallerPC"}:                     struct{}{},
	{"arm", "internal/runtime/sys", "GetCallerSP"}:                     struct{}{},
	{"arm", "internal/runtime/sys", "GetClosurePtr"}:                   struct{}{},
	{"arm", "internal/runtime/sys", "Len64"}:                           struct{}{},
	{"arm", "internal/runtime/sys", "Len8"}:                            struct{}{},
	{"arm", "internal/runtime/sys", "TrailingZeros32"}:                 struct{}{},
	{"arm", "internal/runtime/sys", "TrailingZeros64"}:                 struct{}{},
	{"arm", "internal/runtime/sys", "TrailingZeros8"}:                  struct{}{},
	{"arm", "math", "Abs"}:                                             struct{}{},
	{"arm", "math", "FMA"}:                                             struct{}{},
	{"arm", "math", "sqrt"}:                                            struct{}{},
	{"arm", "math/bits", "Len"}:                                        struct{}{},
	{"arm", "math/bits", "Len16"}:                                      struct{}{},
	{"arm", "math/bits", "Len32"}:                                      struct{}{},
	{"arm", "math/bits", "Len64"}:                                      struct{}{},
	{"arm", "math/bits", "Len8"}:                                       struct{}{},
	{"arm", "math/bits", "ReverseBytes32"}:                             struct{}{},
	{"arm", "math/bits", "ReverseBytes64"}:                             struct{}{},
	{"arm", "math/bits", "RotateLeft32"}:                               struct{}{},
	{"arm", "math/bits", "TrailingZeros16"}:                            struct{}{},
	{"arm", "math/bits", "TrailingZeros32"}:                            struct{}{},
	{"arm", "math/bits", "TrailingZeros64"}:                            struct{}{},
	{"arm", "math/bits", "TrailingZeros8"}:                             struct{}{},
	{"arm", "runtime", "KeepAlive"}:                                    struct{}{},
	{"arm", "runtime", "slicebytetostringtmp"}:                         struct{}{},
	{"arm64", "internal/runtime/atomic", "And"}:                        struct{}{},
	{"arm64", "internal/runtime/atomic", "And32"}:                      struct{}{},
	{"arm64", "internal/runtime/atomic", "And64"}:                      struct{}{},
	{"arm64", "internal/runtime/atomic", "And8"}:                       struct{}{},
	{"arm64", "internal/runtime/atomic", "Anduintptr"}:                 struct{}{},
	{"arm64", "internal/runtime/atomic", "Cas"}:                        struct{}{},
	{"arm64", "internal/runtime/atomic", "Cas64"}:                      struct{}{},
	{"arm64", "internal/runtime/atomic", "CasRel"}:                     struct{}{},
	{"arm64", "internal/runtime/atomic", "Casint32"}:                   struct{}{},
	{"arm64", "internal/runtime/atomic", "Casint64"}:                   struct{}{},
	{"arm64", "internal/runtime/atomic", "Casp1"}:                      struct{}{},
	{"arm64", "internal/runtime/atomic", "Casuintptr"}:                 struct{}{},
	{"arm64", "internal/runtime/atomic", "Load"}:                       struct{}{},
	{"arm64", "internal/runtime/atomic", "Load64"}:                     struct{}{},
	{"arm64", "internal/runtime/atomic", "Load8"}:                      struct{}{},
	{"arm64", "internal/runtime/atomic", "LoadAcq"}:                    struct{}{},
	{"arm64", "internal/runtime/atomic", "LoadAcq64"}:                  struct{}{},
	{"arm64", "internal/runtime/atomic", "LoadAcquintptr"}:             struct{}{},
	{"arm64", "internal/runtime/atomic", "Loadint32"}:                  struct{}{},
	{"arm64", "internal/runtime/atomic", "Loadint64"}:                  struct{}{},
	{"arm64", "internal/runtime/atomic", "Loadp"}:                      struct{}{},
	{"arm64", "internal/runtime/atomic", "Loaduint"}:                   struct{}{},
	{"arm64", "internal/runtime/atomic", "Loaduintptr"}:                struct{}{},
	{"arm64", "internal/runtime/atomic", "Or"}:                         struct{}{},
	{"arm64", "internal/runtime/atomic", "Or32"}:                       struct{}{},
	{"arm64", "internal/runtime/atomic", "Or64"}:                       struct{}{},
	{"arm64", "internal/runtime/atomic", "Or8"}:                        struct{}{},
	{"arm64", "internal/runtime/atomic", "Oruintptr"}:                  struct{}{},
	{"arm64", "internal/runtime/atomic", "Store"}:                      struct{}{},
	{"arm64", "internal/runtime/atomic", "Store64"}:                    struct{}{},
	{"arm64", "internal/runtime/atomic", "Store8"}:                     struct{}{},
	{"arm64", "internal/runtime/atomic", "StoreRel"}:                   struct{}{},
	{"arm64", "internal/runtime/atomic", "StoreRel64"}:                 struct{}{},
	{"arm64", "internal/runtime/atomic", "StoreReluintptr"}:            struct{}{},
	{"arm64", "internal/runtime/atomic", "Storeint32"}:                 struct{}{},
	{"arm64", "internal/runtime/atomic", "Storeint64"}:                 struct{}{},
	{"arm64", "internal/runtime/atomic", "StorepNoWB"}:                 struct{}{},
	{"arm64", "internal/runtime/atomic", "Storeuintptr"}:               struct{}{},
	{"arm64", "internal/runtime/atomic", "Xadd"}:                       struct{}{},
	{"arm64", "internal/runtime/atomic", "Xadd64"}:                     struct{}{},
	{"arm64", "internal/runtime/atomic", "Xaddint32"}:                  struct{}{},
	{"arm64", "internal/runtime/atomic", "Xaddint64"}:                  struct{}{},
	{"arm64", "internal/runtime/atomic", "Xadduintptr"}:                struct{}{},
	{"arm64", "internal/runtime/atomic", "Xchg8"}:                      struct{}{},
	{"arm64", "internal/runtime/atomic", "Xchg"}:                       struct{}{},
	{"arm64", "internal/runtime/atomic", "Xchg64"}:                     struct{}{},
	{"arm64", "internal/runtime/atomic", "Xchgint32"}:                  struct{}{},
	{"arm64", "internal/runtime/atomic", "Xchgint64"}:                  struct{}{},
	{"arm64", "internal/runtime/atomic", "Xchguintptr"}:                struct{}{},
	{"arm64", "internal/runtime/math", "Add64"}:                        struct{}{},
	{"arm64", "internal/runtime/math", "Mul64"}:                        struct{}{},
	{"arm64", "internal/runtime/math", "MulUintptr"}:                   struct{}{},
	{"arm64", "internal/runtime/sys", "Bswap32"}:                       struct{}{},
	{"arm64", "internal/runtime/sys", "Bswap64"}:                       struct{}{},
	{"arm64", "internal/runtime/sys", "GetCallerPC"}:                   struct{}{},
	{"arm64", "internal/runtime/sys", "GetCallerSP"}:                   struct{}{},
	{"arm64", "internal/runtime/sys", "GetClosurePtr"}:                 struct{}{},
	{"arm64", "internal/runtime/sys", "Len64"}:                         struct{}{},
	{"arm64", "internal/runtime/sys", "Len8"}:                          struct{}{},
	{"arm64", "internal/runtime/sys", "OnesCount64"}:                   struct{}{},
	{"arm64", "internal/runtime/sys", "Prefetch"}:                      struct{}{},
	{"arm64", "internal/runtime/sys", "PrefetchStreamed"}:              struct{}{},
	{"arm64", "internal/runtime/sys", "TrailingZeros32"}:               struct{}{},
	{"arm64", "internal/runtime/sys", "TrailingZeros64"}:               struct{}{},
	{"arm64", "internal/runtime/sys", "TrailingZeros8"}:                struct{}{},
	{"arm64", "math", "Abs"}:                                           struct{}{},
	{"arm64", "math", "Ceil"}:                                          struct{}{},
	{"arm64", "math", "FMA"}:                                           struct{}{},
	{"arm64", "math", "Floor"}:                                         struct{}{},
	{"arm64", "math", "Round"}:                                         struct{}{},
	{"arm64", "math", "RoundToEven"}:                                   struct{}{},
	{"arm64", "math", "Trunc"}:                                         struct{}{},
	{"arm64", "math", "sqrt"}:                                          struct{}{},
	{"arm64", "math/big", "mulWW"}:                                     struct{}{},
	{"arm64", "math/bits", "Add"}:                                      struct{}{},
	{"arm64", "math/bits", "Add64"}:                                    struct{}{},
	{"arm64", "math/bits", "Len"}:                                      struct{}{},
	{"arm64", "math/bits", "Len16"}:                                    struct{}{},
	{"arm64", "math/bits", "Len32"}:                                    struct{}{},
	{"arm64", "math/bits", "Len64"}:                                    struct{}{},
	{"arm64", "math/bits", "Len8"}:                                     struct{}{},
	{"arm64", "math/bits", "Mul"}:                                      struct{}{},
	{"arm64", "math/bits", "Mul64"}:                                    struct{}{},
	{"arm64", "math/bits", "OnesCount"}:                                struct{}{},
	{"arm64", "math/bits", "OnesCount16"}:                              struct{}{},
	{"arm64", "math/bits", "OnesCount32"}:                              struct{}{},
	{"arm64", "math/bits", "OnesCount64"}:                              struct{}{},
	{"arm64", "math/bits", "Reverse"}:                                  struct{}{},
	{"arm64", "math/bits", "Reverse16"}:                                struct{}{},
	{"arm64", "math/bits", "Reverse32"}:                                struct{}{},
	{"arm64", "math/bits", "Reverse64"}:                                struct{}{},
	{"arm64", "math/bits", "Reverse8"}:                                 struct{}{},
	{"arm64", "math/bits", "ReverseBytes32"}:                           struct{}{},
	{"arm64", "math/bits", "ReverseBytes64"}:                           struct{}{},
	{"arm64", "math/bits", "RotateLeft"}:                               struct{}{},
	{"arm64", "math/bits", "RotateLeft32"}:                             struct{}{},
	{"arm64", "math/bits", "RotateLeft64"}:                             struct{}{},
	{"arm64", "math/bits", "Sub"}:                                      struct{}{},
	{"arm64", "math/bits", "Sub64"}:                                    struct{}{},
	{"arm64", "math/bits", "TrailingZeros16"}:                          struct{}{},
	{"arm64", "math/bits", "TrailingZeros32"}:                          struct{}{},
	{"arm64", "math/bits", "TrailingZeros64"}:                          struct{}{},
	{"arm64", "math/bits", "TrailingZeros8"}:                           struct{}{},
	{"arm64", "runtime", "KeepAlive"}:                                  struct{}{},
	{"arm64", "runtime", "publicationBarrier"}:                         struct{}{},
	{"arm64", "runtime", "slicebytetostringtmp"}:                       struct{}{},
	{"arm64", "sync", "runtime_LoadAcquintptr"}:                        struct{}{},
	{"arm64", "sync", "runtime_StoreReluintptr"}:                       struct{}{},
	{"arm64", "sync/atomic", "AddInt32"}:                               struct{}{},
	{"arm64", "sync/atomic", "AddInt64"}:                               struct{}{},
	{"arm64", "sync/atomic", "AddUint32"}:                              struct{}{},
	{"arm64", "sync/atomic", "AddUint64"}:                              struct{}{},
	{"arm64", "sync/atomic", "AddUintptr"}:                             struct{}{},
	{"arm64", "sync/atomic", "AndInt32"}:                               struct{}{},
	{"arm64", "sync/atomic", "AndInt64"}:                               struct{}{},
	{"arm64", "sync/atomic", "AndUint32"}:                              struct{}{},
	{"arm64", "sync/atomic", "AndUint64"}:                              struct{}{},
	{"arm64", "sync/atomic", "AndUintptr"}:                             struct{}{},
	{"arm64", "sync/atomic", "CompareAndSwapInt32"}:                    struct{}{},
	{"arm64", "sync/atomic", "CompareAndSwapInt64"}:                    struct{}{},
	{"arm64", "sync/atomic", "CompareAndSwapUint32"}:                   struct{}{},
	{"arm64", "sync/atomic", "CompareAndSwapUint64"}:                   struct{}{},
	{"arm64", "sync/atomic", "CompareAndSwapUintptr"}:                  struct{}{},
	{"arm64", "sync/atomic", "LoadInt32"}:                              struct{}{},
	{"arm64", "sync/atomic", "LoadInt64"}:                              struct{}{},
	{"arm64", "sync/atomic", "LoadPointer"}:                            struct{}{},
	{"arm64", "sync/atomic", "LoadUint32"}:                             struct{}{},
	{"arm64", "sync/atomic", "LoadUint64"}:                             struct{}{},
	{"arm64", "sync/atomic", "LoadUintptr"}:                            struct{}{},
	{"arm64", "sync/atomic", "OrInt32"}:                                struct{}{},
	{"arm64", "sync/atomic", "OrInt64"}:                                struct{}{},
	{"arm64", "sync/atomic", "OrUint32"}:                               struct{}{},
	{"arm64", "sync/atomic", "OrUint64"}:                               struct{}{},
	{"arm64", "sync/atomic", "OrUintptr"}:                              struct{}{},
	{"arm64", "sync/atomic", "StoreInt32"}:                             struct{}{},
	{"arm64", "sync/atomic", "StoreInt64"}:                             struct{}{},
	{"arm64", "sync/atomic", "StoreUint32"}:                            struct{}{},
	{"arm64", "sync/atomic", "StoreUint64"}:                            struct{}{},
	{"arm64", "sync/atomic", "StoreUintptr"}:                           struct{}{},
	{"arm64", "sync/atomic", "SwapInt32"}:                              struct{}{},
	{"arm64", "sync/atomic", "SwapInt64"}:                              struct{}{},
	{"arm64", "sync/atomic", "SwapUint32"}:                             struct{}{},
	{"arm64", "sync/atomic", "SwapUint64"}:                             struct{}{},
	{"arm64", "sync/atomic", "SwapUintptr"}:                            struct{}{},
	{"loong64", "internal/runtime/atomic", "And"}:                      struct{}{},
	{"loong64", "internal/runtime/atomic", "And32"}:                    struct{}{},
	{"loong64", "internal/runtime/atomic", "And64"}:                    struct{}{},
	{"loong64", "internal/runtime/atomic", "And8"}:                     struct{}{},
	{"loong64", "internal/runtime/atomic", "Anduintptr"}:               struct{}{},
	{"loong64", "internal/runtime/atomic", "Cas"}:                      struct{}{},
	{"loong64", "internal/runtime/atomic", "Cas64"}:                    struct{}{},
	{"loong64", "internal/runtime/atomic", "CasRel"}:                   struct{}{},
	{"loong64", "internal/runtime/atomic", "Casint32"}:                 struct{}{},
	{"loong64", "internal/runtime/atomic", "Casint64"}:                 struct{}{},
	{"loong64", "internal/runtime/atomic", "Casp1"}:                    struct{}{},
	{"loong64", "internal/runtime/atomic", "Casuintptr"}:               struct{}{},
	{"loong64", "internal/runtime/atomic", "Load"}:                     struct{}{},
	{"loong64", "internal/runtime/atomic", "Load64"}:                   struct{}{},
	{"loong64", "internal/runtime/atomic", "Load8"}:                    struct{}{},
	{"loong64", "internal/runtime/atomic", "LoadAcq"}:                  struct{}{},
	{"loong64", "internal/runtime/atomic", "LoadAcq64"}:                struct{}{},
	{"loong64", "internal/runtime/atomic", "LoadAcquintptr"}:           struct{}{},
	{"loong64", "internal/runtime/atomic", "Loadint32"}:                struct{}{},
	{"loong64", "internal/runtime/atomic", "Loadint64"}:                struct{}{},
	{"loong64", "internal/runtime/atomic", "Loadp"}:                    struct{}{},
	{"loong64", "internal/runtime/atomic", "Loaduint"}:                 struct{}{},
	{"loong64", "internal/runtime/atomic", "Loaduintptr"}:              struct{}{},
	{"loong64", "internal/runtime/atomic", "Or"}:                       struct{}{},
	{"loong64", "internal/runtime/atomic", "Or32"}:                     struct{}{},
	{"loong64", "internal/runtime/atomic", "Or64"}:                     struct{}{},
	{"loong64", "internal/runtime/atomic", "Or8"}:                      struct{}{},
	{"loong64", "internal/runtime/atomic", "Oruintptr"}:                struct{}{},
	{"loong64", "internal/runtime/atomic", "Store"}:                    struct{}{},
	{"loong64", "internal/runtime/atomic", "Store64"}:                  struct{}{},
	{"loong64", "internal/runtime/atomic", "Store8"}:                   struct{}{},
	{"loong64", "internal/runtime/atomic", "StoreRel"}:                 struct{}{},
	{"loong64", "internal/runtime/atomic", "StoreRel64"}:               struct{}{},
	{"loong64", "internal/runtime/atomic", "StoreReluintptr"}:          struct{}{},
	{"loong64", "internal/runtime/atomic", "Storeint32"}:               struct{}{},
	{"loong64", "internal/runtime/atomic", "Storeint64"}:               struct{}{},
	{"loong64", "internal/runtime/atomic", "StorepNoWB"}:               struct{}{},
	{"loong64", "internal/runtime/atomic", "Storeuintptr"}:             struct{}{},
	{"loong64", "internal/runtime/atomic", "Xadd"}:                     struct{}{},
	{"loong64", "internal/runtime/atomic", "Xadd64"}:                   struct{}{},
	{"loong64", "internal/runtime/atomic", "Xaddint32"}:                struct{}{},
	{"loong64", "internal/runtime/atomic", "Xaddint64"}:                struct{}{},
	{"loong64", "internal/runtime/atomic", "Xadduintptr"}:              struct{}{},
	{"loong64", "internal/runtime/atomic", "Xchg8"}:                    struct{}{},
	{"loong64", "internal/runtime/atomic", "Xchg"}:                     struct{}{},
	{"loong64", "internal/runtime/atomic", "Xchg64"}:                   struct{}{},
	{"loong64", "internal/runtime/atomic", "Xchgint32"}:                struct{}{},
	{"loong64", "internal/runtime/atomic", "Xchgint64"}:                struct{}{},
	{"loong64", "internal/runtime/atomic", "Xchguintptr"}:              struct{}{},
	{"loong64", "internal/runtime/math", "Add64"}:                      struct{}{},
	{"loong64", "internal/runtime/math", "Mul64"}:                      struct{}{},
	{"loong64", "internal/runtime/math", "MulUintptr"}:                 struct{}{},
	{"loong64", "internal/runtime/sys", "Bswap32"}:                     struct{}{},
	{"loong64", "internal/runtime/sys", "Bswap64"}:                     struct{}{},
	{"loong64", "internal/runtime/sys", "GetCallerPC"}:                 struct{}{},
	{"loong64", "internal/runtime/sys", "GetCallerSP"}:                 struct{}{},
	{"loong64", "internal/runtime/sys", "GetClosurePtr"}:               struct{}{},
	{"loong64", "internal/runtime/sys", "Len64"}:                       struct{}{},
	{"loong64", "internal/runtime/sys", "Len8"}:                        struct{}{},
	{"loong64", "internal/runtime/sys", "OnesCount64"}:                 struct{}{},
	{"loong64", "internal/runtime/sys", "Prefetch"}:                    struct{}{},
	{"loong64", "internal/runtime/sys", "PrefetchStreamed"}:            struct{}{},
	{"loong64", "internal/runtime/sys", "TrailingZeros32"}:             struct{}{},
	{"loong64", "internal/runtime/sys", "TrailingZeros64"}:             struct{}{},
	{"loong64", "internal/runtime/sys", "TrailingZeros8"}:              struct{}{},
	{"loong64", "math", "Abs"}:                                         struct{}{},
	{"loong64", "math", "Copysign"}:                                    struct{}{},
	{"loong64", "math", "FMA"}:                                         struct{}{},
	{"loong64", "math", "sqrt"}:                                        struct{}{},
	{"loong64", "math/big", "mulWW"}:                                   struct{}{},
	{"loong64", "math/bits", "Add"}:                                    struct{}{},
	{"loong64", "math/bits", "Add64"}:                                  struct{}{},
	{"loong64", "math/bits", "Mul"}:                                    struct{}{},
	{"loong64", "math/bits", "Mul64"}:                                  struct{}{},
	{"loong64", "math/bits", "Len"}:                                    struct{}{},
	{"loong64", "math/bits", "Len8"}:                                   struct{}{},
	{"loong64", "math/bits", "Len16"}:                                  struct{}{},
	{"loong64", "math/bits", "Len32"}:                                  struct{}{},
	{"loong64", "math/bits", "Len64"}:                                  struct{}{},
	{"loong64", "math/bits", "OnesCount"}:                              struct{}{},
	{"loong64", "math/bits", "OnesCount16"}:                            struct{}{},
	{"loong64", "math/bits", "OnesCount32"}:                            struct{}{},
	{"loong64", "math/bits", "OnesCount64"}:                            struct{}{},
	{"loong64", "math/bits", "Reverse"}:                                struct{}{},
	{"loong64", "math/bits", "Reverse8"}:                               struct{}{},
	{"loong64", "math/bits", "Reverse16"}:                              struct{}{},
	{"loong64", "math/bits", "Reverse32"}:                              struct{}{},
	{"loong64", "math/bits", "Reverse64"}:                              struct{}{},
	{"loong64", "math/bits", "RotateLeft"}:                             struct{}{},
	{"loong64", "math/bits", "RotateLeft32"}:                           struct{}{},
	{"loong64", "math/bits", "RotateLeft64"}:                           struct{}{},
	{"loong64", "math/bits", "ReverseBytes16"}:                         struct{}{},
	{"loong64", "math/bits", "ReverseBytes32"}:                         struct{}{},
	{"loong64", "math/bits", "ReverseBytes64"}:                         struct{}{},
	{"loong64", "math/bits", "TrailingZeros16"}:                        struct{}{},
	{"loong64", "math/bits", "TrailingZeros32"}:                        struct{}{},
	{"loong64", "math/bits", "TrailingZeros64"}:                        struct{}{},
	{"loong64", "math/bits", "TrailingZeros8"}:                         struct{}{},
	{"loong64", "math/bits", "Sub"}:                                    struct{}{},
	{"loong64", "math/bits", "Sub64"}:                                  struct{}{},
	{"loong64", "runtime", "KeepAlive"}:                                struct{}{},
	{"loong64", "runtime", "publicationBarrier"}:                       struct{}{},
	{"loong64", "runtime", "slicebytetostringtmp"}:                     struct{}{},
	{"loong64", "sync", "runtime_LoadAcquintptr"}:                      struct{}{},
	{"loong64", "sync", "runtime_StoreReluintptr"}:                     struct{}{},
	{"loong64", "sync/atomic", "AddInt32"}:                             struct{}{},
	{"loong64", "sync/atomic", "AddInt64"}:                             struct{}{},
	{"loong64", "sync/atomic", "AddUint32"}:                            struct{}{},
	{"loong64", "sync/atomic", "AddUint64"}:                            struct{}{},
	{"loong64", "sync/atomic", "AddUintptr"}:                           struct{}{},
	{"loong64", "sync/atomic", "AddInt32"}:                             struct{}{},
	{"loong64", "sync/atomic", "AddInt64"}:                             struct{}{},
	{"loong64", "sync/atomic", "AddUint32"}:                            struct{}{},
	{"loong64", "sync/atomic", "AddUint64"}:                            struct{}{},
	{"loong64", "sync/atomic", "AddUintptr"}:                           struct{}{},
	{"loong64", "sync/atomic", "AndInt32"}:                             struct{}{},
	{"loong64", "sync/atomic", "AndInt64"}:                             struct{}{},
	{"loong64", "sync/atomic", "AndUint32"}:                            struct{}{},
	{"loong64", "sync/atomic", "AndUint64"}:                            struct{}{},
	{"loong64", "sync/atomic", "AndUintptr"}:                           struct{}{},
	{"loong64", "sync/atomic", "CompareAndSwapInt32"}:                  struct{}{},
	{"loong64", "sync/atomic", "CompareAndSwapInt64"}:                  struct{}{},
	{"loong64", "sync/atomic", "CompareAndSwapUint32"}:                 struct{}{},
	{"loong64", "sync/atomic", "CompareAndSwapUint64"}:                 struct{}{},
	{"loong64", "sync/atomic", "CompareAndSwapUintptr"}:                struct{}{},
	{"loong64", "sync/atomic", "LoadInt32"}:                            struct{}{},
	{"loong64", "sync/atomic", "LoadInt64"}:                            struct{}{},
	{"loong64", "sync/atomic", "LoadPointer"}:                          struct{}{},
	{"loong64", "sync/atomic", "LoadUint32"}:                           struct{}{},
	{"loong64", "sync/atomic", "LoadUint64"}:                           struct{}{},
	{"loong64", "sync/atomic", "LoadUintptr"}:                          struct{}{},
	{"loong64", "sync/atomic", "OrInt32"}:                              struct{}{},
	{"loong64", "sync/atomic", "OrInt64"}:                              struct{}{},
	{"loong64", "sync/atomic", "OrUint32"}:                             struct{}{},
	{"loong64", "sync/atomic", "OrUint64"}:                             struct{}{},
	{"loong64", "sync/atomic", "OrUintptr"}:                            struct{}{},
	{"loong64", "sync/atomic", "StoreInt32"}:                           struct{}{},
	{"loong64", "sync/atomic", "StoreInt64"}:                           struct{}{},
	{"loong64", "sync/atomic", "StoreUint32"}:                          struct{}{},
	{"loong64", "sync/atomic", "StoreUint64"}:                          struct{}{},
	{"loong64", "sync/atomic", "StoreUintptr"}:                         struct{}{},
	{"loong64", "sync/atomic", "SwapInt32"}:                            struct{}{},
	{"loong64", "sync/atomic", "SwapInt64"}:                            struct{}{},
	{"loong64", "sync/atomic", "SwapUint32"}:                           struct{}{},
	{"loong64", "sync/atomic", "SwapUint64"}:                           struct{}{},
	{"loong64", "sync/atomic", "SwapUintptr"}:                          struct{}{},
	{"mips", "internal/runtime/atomic", "And"}:                         struct{}{},
	{"mips", "internal/runtime/atomic", "And8"}:                        struct{}{},
	{"mips", "internal/runtime/atomic", "Cas"}:                         struct{}{},
	{"mips", "internal/runtime/atomic", "CasRel"}:                      struct{}{},
	{"mips", "internal/runtime/atomic", "Casint32"}:                    struct{}{},
	{"mips", "internal/runtime/atomic", "Casp1"}:                       struct{}{},
	{"mips", "internal/runtime/atomic", "Casuintptr"}:                  struct{}{},
	{"mips", "internal/runtime/atomic", "Load"}:                        struct{}{},
	{"mips", "internal/runtime/atomic", "Load8"}:                       struct{}{},
	{"mips", "internal/runtime/atomic", "LoadAcq"}:                     struct{}{},
	{"mips", "internal/runtime/atomic", "LoadAcquintptr"}:              struct{}{},
	{"mips", "internal/runtime/atomic", "Loadint32"}:                   struct{}{},
	{"mips", "internal/runtime/atomic", "Loadp"}:                       struct{}{},
	{"mips", "internal/runtime/atomic", "Loaduint"}:                    struct{}{},
	{"mips", "internal/runtime/atomic", "Loaduintptr"}:                 struct{}{},
	{"mips", "internal/runtime/atomic", "Or"}:                          struct{}{},
	{"mips", "internal/runtime/atomic", "Or8"}:                         struct{}{},
	{"mips", "internal/runtime/atomic", "Store"}:                       struct{}{},
	{"mips", "internal/runtime/atomic", "Store8"}:                      struct{}{},
	{"mips", "internal/runtime/atomic", "StoreRel"}:                    struct{}{},
	{"mips", "internal/runtime/atomic", "StoreReluintptr"}:             struct{}{},
	{"mips", "internal/runtime/atomic", "Storeint32"}:                  struct{}{},
	{"mips", "internal/runtime/atomic", "StorepNoWB"}:                  struct{}{},
	{"mips", "internal/runtime/atomic", "Storeuintptr"}:                struct{}{},
	{"mips", "internal/runtime/atomic", "Xadd"}:                        struct{}{},
	{"mips", "internal/runtime/atomic", "Xaddint32"}:                   struct{}{},
	{"mips", "internal/runtime/atomic", "Xadduintptr"}:                 struct{}{},
	{"mips", "internal/runtime/atomic", "Xchg"}:                        struct{}{},
	{"mips", "internal/runtime/atomic", "Xchgint32"}:                   struct{}{},
	{"mips", "internal/runtime/atomic", "Xchguintptr"}:                 struct{}{},
	{"mips", "internal/runtime/sys", "GetCallerPC"}:                    struct{}{},
	{"mips", "internal/runtime/sys", "GetCallerSP"}:                    struct{}{},
	{"mips", "internal/runtime/sys", "GetClosurePtr"}:                  struct{}{},
	{"mips", "internal/runtime/sys", "Len64"}:                          struct{}{},
	{"mips", "internal/runtime/sys", "Len8"}:                           struct{}{},
	{"mips", "internal/runtime/sys", "TrailingZeros32"}:                struct{}{},
	{"mips", "internal/runtime/sys", "TrailingZeros64"}:                struct{}{},
	{"mips", "internal/runtime/sys", "TrailingZeros8"}:                 struct{}{},
	{"mips", "math", "Abs"}:                                            struct{}{},
	{"mips", "math", "sqrt"}:                                           struct{}{},
	{"mips", "math/bits", "Len"}:                                       struct{}{},
	{"mips", "math/bits", "Len16"}:                                     struct{}{},
	{"mips", "math/bits", "Len32"}:                                     struct{}{},
	{"mips", "math/bits", "Len64"}:                                     struct{}{},
	{"mips", "math/bits", "Len8"}:                                      struct{}{},
	{"mips", "math/bits", "TrailingZeros16"}:                           struct{}{},
	{"mips", "math/bits", "TrailingZeros32"}:                           struct{}{},
	{"mips", "math/bits", "TrailingZeros64"}:                           struct{}{},
	{"mips", "math/bits", "TrailingZeros8"}:                            struct{}{},
	{"mips", "runtime", "KeepAlive"}:                                   struct{}{},
	{"mips", "runtime", "publicationBarrier"}:                          struct{}{},
	{"mips", "runtime", "slicebytetostringtmp"}:                        struct{}{},
	{"mips", "sync", "runtime_LoadAcquintptr"}:                         struct{}{},
	{"mips", "sync", "runtime_StoreReluintptr"}:                        struct{}{},
	{"mips", "sync/atomic", "AddInt32"}:                                struct{}{},
	{"mips", "sync/atomic", "AddUint32"}:                               struct{}{},
	{"mips", "sync/atomic", "AddUintptr"}:                              struct{}{},
	{"mips", "sync/atomic", "CompareAndSwapInt32"}:                     struct{}{},
	{"mips", "sync/atomic", "CompareAndSwapUint32"}:                    struct{}{},
	{"mips", "sync/atomic", "CompareAndSwapUintptr"}:                   struct{}{},
	{"mips", "sync/atomic", "LoadInt32"}:                               struct{}{},
	{"mips", "sync/atomic", "LoadPointer"}:                             struct{}{},
	{"mips", "sync/atomic", "LoadUint32"}:                              struct{}{},
	{"mips", "sync/atomic", "LoadUintptr"}:                             struct{}{},
	{"mips", "sync/atomic", "StoreInt32"}:                              struct{}{},
	{"mips", "sync/atomic", "StoreUint32"}:                             struct{}{},
	{"mips", "sync/atomic", "StoreUintptr"}:                            struct{}{},
	{"mips", "sync/atomic", "SwapInt32"}:                               struct{}{},
	{"mips", "sync/atomic", "SwapUint32"}:                              struct{}{},
	{"mips", "sync/atomic", "SwapUintptr"}:                             struct{}{},
	{"mips64", "internal/runtime/atomic", "And"}:                       struct{}{},
	{"mips64", "internal/runtime/atomic", "And8"}:                      struct{}{},
	{"mips64", "internal/runtime/atomic", "Cas"}:                       struct{}{},
	{"mips64", "internal/runtime/atomic", "Cas64"}:                     struct{}{},
	{"mips64", "internal/runtime/atomic", "CasRel"}:                    struct{}{},
	{"mips64", "internal/runtime/atomic", "Casint32"}:                  struct{}{},
	{"mips64", "internal/runtime/atomic", "Casint64"}:                  struct{}{},
	{"mips64", "internal/runtime/atomic", "Casp1"}:                     struct{}{},
	{"mips64", "internal/runtime/atomic", "Casuintptr"}:                struct{}{},
	{"mips64", "internal/runtime/atomic", "Load"}:                      struct{}{},
	{"mips64", "internal/runtime/atomic", "Load64"}:                    struct{}{},
	{"mips64", "internal/runtime/atomic", "Load8"}:                     struct{}{},
	{"mips64", "internal/runtime/atomic", "LoadAcq"}:                   struct{}{},
	{"mips64", "internal/runtime/atomic", "LoadAcq64"}:                 struct{}{},
	{"mips64", "internal/runtime/atomic", "LoadAcquintptr"}:            struct{}{},
	{"mips64", "internal/runtime/atomic", "Loadint32"}:                 struct{}{},
	{"mips64", "internal/runtime/atomic", "Loadint64"}:                 struct{}{},
	{"mips64", "internal/runtime/atomic", "Loadp"}:                     struct{}{},
	{"mips64", "internal/runtime/atomic", "Loaduint"}:                  struct{}{},
	{"mips64", "internal/runtime/atomic", "Loaduintptr"}:               struct{}{},
	{"mips64", "internal/runtime/atomic", "Or"}:                        struct{}{},
	{"mips64", "internal/runtime/atomic", "Or8"}:                       struct{}{},
	{"mips64", "internal/runtime/atomic", "Store"}:                     struct{}{},
	{"mips64", "internal/runtime/atomic", "Store64"}:                   struct{}{},
	{"mips64", "internal/runtime/atomic", "Store8"}:                    struct{}{},
	{"mips64", "internal/runtime/atomic", "StoreRel"}:                  struct{}{},
	{"mips64", "internal/runtime/atomic", "StoreRel64"}:                struct{}{},
	{"mips64", "internal/runtime/atomic", "StoreReluintptr"}:           struct{}{},
	{"mips64", "internal/runtime/atomic", "Storeint32"}:                struct{}{},
	{"mips64", "internal/runtime/atomic", "Storeint64"}:                struct{}{},
	{"mips64", "internal/runtime/atomic", "StorepNoWB"}:                struct{}{},
	{"mips64", "internal/runtime/atomic", "Storeuintptr"}:              struct{}{},
	{"mips64", "internal/runtime/atomic", "Xadd"}:                      struct{}{},
	{"mips64", "internal/runtime/atomic", "Xadd64"}:                    struct{}{},
	{"mips64", "internal/runtime/atomic", "Xaddint32"}:                 struct{}{},
	{"mips64", "internal/runtime/atomic", "Xaddint64"}:                 struct{}{},
	{"mips64", "internal/runtime/atomic", "Xadduintptr"}:               struct{}{},
	{"mips64", "internal/runtime/atomic", "Xchg"}:                      struct{}{},
	{"mips64", "internal/runtime/atomic", "Xchg64"}:                    struct{}{},
	{"mips64", "internal/runtime/atomic", "Xchgint32"}:                 struct{}{},
	{"mips64", "internal/runtime/atomic", "Xchgint64"}:                 struct{}{},
	{"mips64", "internal/runtime/atomic", "Xchguintptr"}:               struct{}{},
	{"mips64", "internal/runtime/math", "Add64"}:                       struct{}{},
	{"mips64", "internal/runtime/math", "Mul64"}:                       struct{}{},
	{"mips64", "internal/runtime/math", "MulUintptr"}:                  struct{}{},
	{"mips64", "internal/runtime/sys", "GetCallerPC"}:                  struct{}{},
	{"mips64", "internal/runtime/sys", "GetCallerSP"}:                  struct{}{},
	{"mips64", "internal/runtime/sys", "GetClosurePtr"}:                struct{}{},
	{"mips64", "math", "Abs"}:                                          struct{}{},
	{"mips64", "math", "sqrt"}:                                         struct{}{},
	{"mips64", "math/big", "mulWW"}:                                    struct{}{},
	{"mips64", "math/bits", "Add"}:                                     struct{}{},
	{"mips64", "math/bits", "Add64"}:                                   struct{}{},
	{"mips64", "math/bits", "Mul"}:                                     struct{}{},
	{"mips64", "math/bits", "Mul64"}:                                   struct{}{},
	{"mips64", "math/bits", "Sub"}:                                     struct{}{},
	{"mips64", "math/bits", "Sub64"}:                                   struct{}{},
	{"mips64", "runtime", "KeepAlive"}:                                 struct{}{},
	{"mips64", "runtime", "publicationBarrier"}:                        struct{}{},
	{"mips64", "runtime", "slicebytetostringtmp"}:                      struct{}{},
	{"mips64", "sync", "runtime_LoadAcquintptr"}:                       struct{}{},
	{"mips64", "sync", "runtime_StoreReluintptr"}:                      struct{}{},
	{"mips64", "sync/atomic", "AddInt32"}:                              struct{}{},
	{"mips64", "sync/atomic", "AddInt64"}:                              struct{}{},
	{"mips64", "sync/atomic", "AddUint32"}:                             struct{}{},
	{"mips64", "sync/atomic", "AddUint64"}:                             struct{}{},
	{"mips64", "sync/atomic", "AddUintptr"}:                            struct{}{},
	{"mips64", "sync/atomic", "CompareAndSwapInt32"}:                   struct{}{},
	{"mips64", "sync/atomic", "CompareAndSwapInt64"}:                   struct{}{},
	{"mips64", "sync/atomic", "CompareAndSwapUint32"}:                  struct{}{},
	{"mips64", "sync/atomic", "CompareAndSwapUint64"}:                  struct{}{},
	{"mips64", "sync/atomic", "CompareAndSwapUintptr"}:                 struct{}{},
	{"mips64", "sync/atomic", "LoadInt32"}:                             struct{}{},
	{"mips64", "sync/atomic", "LoadInt64"}:                             struct{}{},
	{"mips64", "sync/atomic", "LoadPointer"}:                           struct{}{},
	{"mips64", "sync/atomic", "LoadUint32"}:                            struct{}{},
	{"mips64", "sync/atomic", "LoadUint64"}:                            struct{}{},
	{"mips64", "sync/atomic", "LoadUintptr"}:                           struct{}{},
	{"mips64", "sync/atomic", "StoreInt32"}:                            struct{}{},
	{"mips64", "sync/atomic", "StoreInt64"}:                            struct{}{},
	{"mips64", "sync/atomic", "StoreUint32"}:                           struct{}{},
	{"mips64", "sync/atomic", "StoreUint64"}:                           struct{}{},
	{"mips64", "sync/atomic", "StoreUintptr"}:                          struct{}{},
	{"mips64", "sync/atomic", "SwapInt32"}:                             struct{}{},
	{"mips64", "sync/atomic", "SwapInt64"}:                             struct{}{},
	{"mips64", "sync/atomic", "SwapUint32"}:                            struct{}{},
	{"mips64", "sync/atomic", "SwapUint64"}:                            struct{}{},
	{"mips64", "sync/atomic", "SwapUintptr"}:                           struct{}{},
	{"mips64le", "internal/runtime/atomic", "And"}:                     struct{}{},
	{"mips64le", "internal/runtime/atomic", "And8"}:                    struct{}{},
	{"mips64le", "internal/runtime/atomic", "Cas"}:                     struct{}{},
	{"mips64le", "internal/runtime/atomic", "Cas64"}:                   struct{}{},
	{"mips64le", "internal/runtime/atomic", "CasRel"}:                  struct{}{},
	{"mips64le", "internal/runtime/atomic", "Casint32"}:                struct{}{},
	{"mips64le", "internal/runtime/atomic", "Casint64"}:                struct{}{},
	{"mips64le", "internal/runtime/atomic", "Casp1"}:                   struct{}{},
	{"mips64le", "internal/runtime/atomic", "Casuintptr"}:              struct{}{},
	{"mips64le", "internal/runtime/atomic", "Load"}:                    struct{}{},
	{"mips64le", "internal/runtime/atomic", "Load64"}:                  struct{}{},
	{"mips64le", "internal/runtime/atomic", "Load8"}:                   struct{}{},
	{"mips64le", "internal/runtime/atomic", "LoadAcq"}:                 struct{}{},
	{"mips64le", "internal/runtime/atomic", "LoadAcq64"}:               struct{}{},
	{"mips64le", "internal/runtime/atomic", "LoadAcquintptr"}:          struct{}{},
	{"mips64le", "internal/runtime/atomic", "Loadint32"}:               struct{}{},
	{"mips64le", "internal/runtime/atomic", "Loadint64"}:               struct{}{},
	{"mips64le", "internal/runtime/atomic", "Loadp"}:                   struct{}{},
	{"mips64le", "internal/runtime/atomic", "Loaduint"}:                struct{}{},
	{"mips64le", "internal/runtime/atomic", "Loaduintptr"}:             struct{}{},
	{"mips64le", "internal/runtime/atomic", "Or"}:                      struct{}{},
	{"mips64le", "internal/runtime/atomic", "Or8"}:                     struct{}{},
	{"mips64le", "internal/runtime/atomic", "Store"}:                   struct{}{},
	{"mips64le", "internal/runtime/atomic", "Store64"}:                 struct{}{},
	{"mips64le", "internal/runtime/atomic", "Store8"}:                  struct{}{},
	{"mips64le", "internal/runtime/atomic", "StoreRel"}:                struct{}{},
	{"mips64le", "internal/runtime/atomic", "StoreRel64"}:              struct{}{},
	{"mips64le", "internal/runtime/atomic", "StoreReluintptr"}:         struct{}{},
	{"mips64le", "internal/runtime/atomic", "Storeint32"}:              struct{}{},
	{"mips64le", "internal/runtime/atomic", "Storeint64"}:              struct{}{},
	{"mips64le", "internal/runtime/atomic", "StorepNoWB"}:              struct{}{},
	{"mips64le", "internal/runtime/atomic", "Storeuintptr"}:            struct{}{},
	{"mips64le", "internal/runtime/atomic", "Xadd"}:                    struct{}{},
	{"mips64le", "internal/runtime/atomic", "Xadd64"}:                  struct{}{},
	{"mips64le", "internal/runtime/atomic", "Xaddint32"}:               struct{}{},
	{"mips64le", "internal/runtime/atomic", "Xaddint64"}:               struct{}{},
	{"mips64le", "internal/runtime/atomic", "Xadduintptr"}:             struct{}{},
	{"mips64le", "internal/runtime/atomic", "Xchg"}:                    struct{}{},
	{"mips64le", "internal/runtime/atomic", "Xchg64"}:                  struct{}{},
	{"mips64le", "internal/runtime/atomic", "Xchgint32"}:               struct{}{},
	{"mips64le", "internal/runtime/atomic", "Xchgint64"}:               struct{}{},
	{"mips64le", "internal/runtime/atomic", "Xchguintptr"}:             struct{}{},
	{"mips64le", "internal/runtime/math", "Add64"}:                     struct{}{},
	{"mips64le", "internal/runtime/math", "Mul64"}:                     struct{}{},
	{"mips64le", "internal/runtime/math", "MulUintptr"}:                struct{}{},
	{"mips64le", "internal/runtime/sys", "GetCallerPC"}:                struct{}{},
	{"mips64le", "internal/runtime/sys", "GetCallerSP"}:                struct{}{},
	{"mips64le", "internal/runtime/sys", "GetClosurePtr"}:              struct{}{},
	{"mips64le", "math", "Abs"}:                                        struct{}{},
	{"mips64le", "math", "sqrt"}:                                       struct{}{},
	{"mips64le", "math/big", "mulWW"}:                                  struct{}{},
	{"mips64le", "math/bits", "Add"}:                                   struct{}{},
	{"mips64le", "math/bits", "Add64"}:                                 struct{}{},
	{"mips64le", "math/bits", "Mul"}:                                   struct{}{},
	{"mips64le", "math/bits", "Mul64"}:                                 struct{}{},
	{"mips64le", "math/bits", "Sub"}:                                   struct{}{},
	{"mips64le", "math/bits", "Sub64"}:                                 struct{}{},
	{"mips64le", "runtime", "KeepAlive"}:                               struct{}{},
	{"mips64le", "runtime", "publicationBarrier"}:                      struct{}{},
	{"mips64le", "runtime", "slicebytetostringtmp"}:                    struct{}{},
	{"mips64le", "sync", "runtime_LoadAcquintptr"}:                     struct{}{},
	{"mips64le", "sync", "runtime_StoreReluintptr"}:                    struct{}{},
	{"mips64le", "sync/atomic", "AddInt32"}:                            struct{}{},
	{"mips64le", "sync/atomic", "AddInt64"}:                            struct{}{},
	{"mips64le", "sync/atomic", "AddUint32"}:                           struct{}{},
	{"mips64le", "sync/atomic", "AddUint64"}:                           struct{}{},
	{"mips64le", "sync/atomic", "AddUintptr"}:                          struct{}{},
	{"mips64le", "sync/atomic", "CompareAndSwapInt32"}:                 struct{}{},
	{"mips64le", "sync/atomic", "CompareAndSwapInt64"}:                 struct{}{},
	{"mips64le", "sync/atomic", "CompareAndSwapUint32"}:                struct{}{},
	{"mips64le", "sync/atomic", "CompareAndSwapUint64"}:                struct{}{},
	{"mips64le", "sync/atomic", "CompareAndSwapUintptr"}:               struct{}{},
	{"mips64le", "sync/atomic", "LoadInt32"}:                           struct{}{},
	{"mips64le", "sync/atomic", "LoadInt64"}:                           struct{}{},
	{"mips64le", "sync/atomic", "LoadPointer"}:                         struct{}{},
	{"mips64le", "sync/atomic", "LoadUint32"}:                          struct{}{},
	{"mips64le", "sync/atomic", "LoadUint64"}:                          struct{}{},
	{"mips64le", "sync/atomic", "LoadUintptr"}:                         struct{}{},
	{"mips64le", "sync/atomic", "StoreInt32"}:                          struct{}{},
	{"mips64le", "sync/atomic", "StoreInt64"}:                          struct{}{},
	{"mips64le", "sync/atomic", "StoreUint32"}:                         struct{}{},
	{"mips64le", "sync/atomic", "StoreUint64"}:                         struct{}{},
	{"mips64le", "sync/atomic", "StoreUintptr"}:                        struct{}{},
	{"mips64le", "sync/atomic", "SwapInt32"}:                           struct{}{},
	{"mips64le", "sync/atomic", "SwapInt64"}:                           struct{}{},
	{"mips64le", "sync/atomic", "SwapUint32"}:                          struct{}{},
	{"mips64le", "sync/atomic", "SwapUint64"}:                          struct{}{},
	{"mips64le", "sync/atomic", "SwapUintptr"}:                         struct{}{},
	{"mipsle", "internal/runtime/atomic", "And"}:                       struct{}{},
	{"mipsle", "internal/runtime/atomic", "And8"}:                      struct{}{},
	{"mipsle", "internal/runtime/atomic", "Cas"}:                       struct{}{},
	{"mipsle", "internal/runtime/atomic", "CasRel"}:                    struct{}{},
	{"mipsle", "internal/runtime/atomic", "Casint32"}:                  struct{}{},
	{"mipsle", "internal/runtime/atomic", "Casp1"}:                     struct{}{},
	{"mipsle", "internal/runtime/atomic", "Casuintptr"}:                struct{}{},
	{"mipsle", "internal/runtime/atomic", "Load"}:                      struct{}{},
	{"mipsle", "internal/runtime/atomic", "Load8"}:                     struct{}{},
	{"mipsle", "internal/runtime/atomic", "LoadAcq"}:                   struct{}{},
	{"mipsle", "internal/runtime/atomic", "LoadAcquintptr"}:            struct{}{},
	{"mipsle", "internal/runtime/atomic", "Loadint32"}:                 struct{}{},
	{"mipsle", "internal/runtime/atomic", "Loadp"}:                     struct{}{},
	{"mipsle", "internal/runtime/atomic", "Loaduint"}:                  struct{}{},
	{"mipsle", "internal/runtime/atomic", "Loaduintptr"}:               struct{}{},
	{"mipsle", "internal/runtime/atomic", "Or"}:                        struct{}{},
	{"mipsle", "internal/runtime/atomic", "Or8"}:                       struct{}{},
	{"mipsle", "internal/runtime/atomic", "Store"}:                     struct{}{},
	{"mipsle", "internal/runtime/atomic", "Store8"}:                    struct{}{},
	{"mipsle", "internal/runtime/atomic", "StoreRel"}:                  struct{}{},
	{"mipsle", "internal/runtime/atomic", "StoreReluintptr"}:           struct{}{},
	{"mipsle", "internal/runtime/atomic", "Storeint32"}:                struct{}{},
	{"mipsle", "internal/runtime/atomic", "StorepNoWB"}:                struct{}{},
	{"mipsle", "internal/runtime/atomic", "Storeuintptr"}:              struct{}{},
	{"mipsle", "internal/runtime/atomic", "Xadd"}:                      struct{}{},
	{"mipsle", "internal/runtime/atomic", "Xaddint32"}:                 struct{}{},
	{"mipsle", "internal/runtime/atomic", "Xadduintptr"}:               struct{}{},
	{"mipsle", "internal/runtime/atomic", "Xchg"}:                      struct{}{},
	{"mipsle", "internal/runtime/atomic", "Xchgint32"}:                 struct{}{},
	{"mipsle", "internal/runtime/atomic", "Xchguintptr"}:               struct{}{},
	{"mipsle", "internal/runtime/sys", "GetCallerPC"}:                  struct{}{},
	{"mipsle", "internal/runtime/sys", "GetCallerSP"}:                  struct{}{},
	{"mipsle", "internal/runtime/sys", "GetClosurePtr"}:                struct{}{},
	{"mipsle", "internal/runtime/sys", "Len64"}:                        struct{}{},
	{"mipsle", "internal/runtime/sys", "Len8"}:                         struct{}{},
	{"mipsle", "internal/runtime/sys", "TrailingZeros32"}:              struct{}{},
	{"mipsle", "internal/runtime/sys", "TrailingZeros64"}:              struct{}{},
	{"mipsle", "internal/runtime/sys", "TrailingZeros8"}:               struct{}{},
	{"mipsle", "math", "Abs"}:                                          struct{}{},
	{"mipsle", "math", "sqrt"}:                                         struct{}{},
	{"mipsle", "math/bits", "Len"}:                                     struct{}{},
	{"mipsle", "math/bits", "Len16"}:                                   struct{}{},
	{"mipsle", "math/bits", "Len32"}:                                   struct{}{},
	{"mipsle", "math/bits", "Len64"}:                                   struct{}{},
	{"mipsle", "math/bits", "Len8"}:                                    struct{}{},
	{"mipsle", "math/bits", "TrailingZeros16"}:                         struct{}{},
	{"mipsle", "math/bits", "TrailingZeros32"}:                         struct{}{},
	{"mipsle", "math/bits", "TrailingZeros64"}:                         struct{}{},
	{"mipsle", "math/bits", "TrailingZeros8"}:                          struct{}{},
	{"mipsle", "runtime", "KeepAlive"}:                                 struct{}{},
	{"mipsle", "runtime", "publicationBarrier"}:                        struct{}{},
	{"mipsle", "runtime", "slicebytetostringtmp"}:                      struct{}{},
	{"mipsle", "sync", "runtime_LoadAcquintptr"}:                       struct{}{},
	{"mipsle", "sync", "runtime_StoreReluintptr"}:                      struct{}{},
	{"mipsle", "sync/atomic", "AddInt32"}:                              struct{}{},
	{"mipsle", "sync/atomic", "AddUint32"}:                             struct{}{},
	{"mipsle", "sync/atomic", "AddUintptr"}:                            struct{}{},
	{"mipsle", "sync/atomic", "CompareAndSwapInt32"}:                   struct{}{},
	{"mipsle", "sync/atomic", "CompareAndSwapUint32"}:                  struct{}{},
	{"mipsle", "sync/atomic", "CompareAndSwapUintptr"}:                 struct{}{},
	{"mipsle", "sync/atomic", "LoadInt32"}:                             struct{}{},
	{"mipsle", "sync/atomic", "LoadPointer"}:                           struct{}{},
	{"mipsle", "sync/atomic", "LoadUint32"}:                            struct{}{},
	{"mipsle", "sync/atomic", "LoadUintptr"}:                           struct{}{},
	{"mipsle", "sync/atomic", "StoreInt32"}:                            struct{}{},
	{"mipsle", "sync/atomic", "StoreUint32"}:                           struct{}{},
	{"mipsle", "sync/atomic", "StoreUintptr"}:                          struct{}{},
	{"mipsle", "sync/atomic", "SwapInt32"}:                             struct{}{},
	{"mipsle", "sync/atomic", "SwapUint32"}:                            struct{}{},
	{"mipsle", "sync/atomic", "SwapUintptr"}:                           struct{}{},
	{"ppc64", "internal/runtime/atomic", "And"}:                        struct{}{},
	{"ppc64", "internal/runtime/atomic", "And8"}:                       struct{}{},
	{"ppc64", "internal/runtime/atomic", "Cas"}:                        struct{}{},
	{"ppc64", "internal/runtime/atomic", "Cas64"}:                      struct{}{},
	{"ppc64", "internal/runtime/atomic", "CasRel"}:                     struct{}{},
	{"ppc64", "internal/runtime/atomic", "Casint32"}:                   struct{}{},
	{"ppc64", "internal/runtime/atomic", "Casint64"}:                   struct{}{},
	{"ppc64", "internal/runtime/atomic", "Casp1"}:                      struct{}{},
	{"ppc64", "internal/runtime/atomic", "Casuintptr"}:                 struct{}{},
	{"ppc64", "internal/runtime/atomic", "Load"}:                       struct{}{},
	{"ppc64", "internal/runtime/atomic", "Load64"}:                     struct{}{},
	{"ppc64", "internal/runtime/atomic", "Load8"}:                      struct{}{},
	{"ppc64", "internal/runtime/atomic", "LoadAcq"}:                    struct{}{},
	{"ppc64", "internal/runtime/atomic", "LoadAcq64"}:                  struct{}{},
	{"ppc64", "internal/runtime/atomic", "LoadAcquintptr"}:             struct{}{},
	{"ppc64", "internal/runtime/atomic", "Loadint32"}:                  struct{}{},
	{"ppc64", "internal/runtime/atomic", "Loadint64"}:                  struct{}{},
	{"ppc64", "internal/runtime/atomic", "Loadp"}:                      struct{}{},
	{"ppc64", "internal/runtime/atomic", "Loaduint"}:                   struct{}{},
	{"ppc64", "internal/runtime/atomic", "Loaduintptr"}:                struct{}{},
	{"ppc64", "internal/runtime/atomic", "Or"}:                         struct{}{},
	{"ppc64", "internal/runtime/atomic", "Or8"}:                        struct{}{},
	{"ppc64", "internal/runtime/atomic", "Store"}:                      struct{}{},
	{"ppc64", "internal/runtime/atomic", "Store64"}:                    struct{}{},
	{"ppc64", "internal/runtime/atomic", "Store8"}:                     struct{}{},
	{"ppc64", "internal/runtime/atomic", "StoreRel"}:                   struct{}{},
	{"ppc64", "internal/runtime/atomic", "StoreRel64"}:                 struct{}{},
	{"ppc64", "internal/runtime/atomic", "StoreReluintptr"}:            struct{}{},
	{"ppc64", "internal/runtime/atomic", "Storeint32"}:                 struct{}{},
	{"ppc64", "internal/runtime/atomic", "Storeint64"}:                 struct{}{},
	{"ppc64", "internal/runtime/atomic", "Storeuintptr"}:               struct{}{},
	{"ppc64", "internal/runtime/atomic", "Xadd"}:                       struct{}{},
	{"ppc64", "internal/runtime/atomic", "Xadd64"}:                     struct{}{},
	{"ppc64", "internal/runtime/atomic", "Xaddint32"}:                  struct{}{},
	{"ppc64", "internal/runtime/atomic", "Xaddint64"}:                  struct{}{},
	{"ppc64", "internal/runtime/atomic", "Xadduintptr"}:                struct{}{},
	{"ppc64", "internal/runtime/atomic", "Xchg8"}:                      struct{}{},
	{"ppc64", "internal/runtime/atomic", "Xchg"}:                       struct{}{},
	{"ppc64", "internal/runtime/atomic", "Xchg64"}:                     struct{}{},
	{"ppc64", "internal/runtime/atomic", "Xchgint32"}:                  struct{}{},
	{"ppc64", "internal/runtime/atomic", "Xchgint64"}:                  struct{}{},
	{"ppc64", "internal/runtime/atomic", "Xchguintptr"}:                struct{}{},
	{"ppc64", "internal/runtime/math", "Add64"}:                        struct{}{},
	{"ppc64", "internal/runtime/math", "Mul64"}:                        struct{}{},
	{"ppc64", "internal/runtime/math", "MulUintptr"}:                   struct{}{},
	{"ppc64", "internal/runtime/sys", "Bswap32"}:                       struct{}{},
	{"ppc64", "internal/runtime/sys", "Bswap64"}:                       struct{}{},
	{"ppc64", "internal/runtime/sys", "GetCallerPC"}:                   struct{}{},
	{"ppc64", "internal/runtime/sys", "GetCallerSP"}:                   struct{}{},
	{"ppc64", "internal/runtime/sys", "GetClosurePtr"}:                 struct{}{},
	{"ppc64", "internal/runtime/sys", "Len64"}:                         struct{}{},
	{"ppc64", "internal/runtime/sys", "Len8"}:                          struct{}{},
	{"ppc64", "internal/runtime/sys", "OnesCount64"}:                   struct{}{},
	{"ppc64", "internal/runtime/sys", "Prefetch"}:                      struct{}{},
	{"ppc64", "internal/runtime/sys", "PrefetchStreamed"}:              struct{}{},
	{"ppc64", "internal/runtime/sys", "TrailingZeros8"}:                struct{}{},
	{"ppc64", "internal/runtime/sys", "TrailingZeros32"}:               struct{}{},
	{"ppc64", "internal/runtime/sys", "TrailingZeros64"}:               struct{}{},
	{"ppc64", "math", "Abs"}:                                           struct{}{},
	{"ppc64", "math", "Ceil"}:                                          struct{}{},
	{"ppc64", "math", "Copysign"}:                                      struct{}{},
	{"ppc64", "math", "FMA"}:                                           struct{}{},
	{"ppc64", "math", "Floor"}:                                         struct{}{},
	{"ppc64", "math", "Round"}:                                         struct{}{},
	{"ppc64", "math", "Trunc"}:                                         struct{}{},
	{"ppc64", "math", "sqrt"}:                                          struct{}{},
	{"ppc64", "math/big", "mulWW"}:                                     struct{}{},
	{"ppc64", "math/bits", "Add"}:                                      struct{}{},
	{"ppc64", "math/bits", "Add64"}:                                    struct{}{},
	{"ppc64", "math/bits", "Len"}:                                      struct{}{},
	{"ppc64", "math/bits", "Len16"}:                                    struct{}{},
	{"ppc64", "math/bits", "Len32"}:                                    struct{}{},
	{"ppc64", "math/bits", "Len64"}:                                    struct{}{},
	{"ppc64", "math/bits", "Len8"}:                                     struct{}{},
	{"ppc64", "math/bits", "Mul"}:                                      struct{}{},
	{"ppc64", "math/bits", "Mul64"}:                                    struct{}{},
	{"ppc64", "math/bits", "OnesCount"}:                                struct{}{},
	{"ppc64", "math/bits", "OnesCount16"}:                              struct{}{},
	{"ppc64", "math/bits", "OnesCount32"}:                              struct{}{},
	{"ppc64", "math/bits", "OnesCount64"}:                              struct{}{},
	{"ppc64", "math/bits", "OnesCount8"}:                               struct{}{},
	{"ppc64", "math/bits", "ReverseBytes16"}:                           struct{}{},
	{"ppc64", "math/bits", "ReverseBytes32"}:                           struct{}{},
	{"ppc64", "math/bits", "ReverseBytes64"}:                           struct{}{},
	{"ppc64", "math/bits", "RotateLeft"}:                               struct{}{},
	{"ppc64", "math/bits", "RotateLeft32"}:                             struct{}{},
	{"ppc64", "math/bits", "RotateLeft64"}:                             struct{}{},
	{"ppc64", "math/bits", "Sub"}:                                      struct{}{},
	{"ppc64", "math/bits", "Sub64"}:                                    struct{}{},
	{"ppc64", "math/bits", "TrailingZeros8"}:                           struct{}{},
	{"ppc64", "math/bits", "TrailingZeros16"}:                          struct{}{},
	{"ppc64", "math/bits", "TrailingZeros32"}:                          struct{}{},
	{"ppc64", "math/bits", "TrailingZeros64"}:                          struct{}{},
	{"ppc64", "runtime", "KeepAlive"}:                                  struct{}{},
	{"ppc64", "runtime", "publicationBarrier"}:                         struct{}{},
	{"ppc64", "runtime", "slicebytetostringtmp"}:                       struct{}{},
	{"ppc64", "sync", "runtime_LoadAcquintptr"}:                        struct{}{},
	{"ppc64", "sync", "runtime_StoreReluintptr"}:                       struct{}{},
	{"ppc64", "sync/atomic", "AddInt32"}:                               struct{}{},
	{"ppc64", "sync/atomic", "AddInt64"}:                               struct{}{},
	{"ppc64", "sync/atomic", "AddUint32"}:                              struct{}{},
	{"ppc64", "sync/atomic", "AddUint64"}:                              struct{}{},
	{"ppc64", "sync/atomic", "AddUintptr"}:                             struct{}{},
	{"ppc64", "sync/atomic", "CompareAndSwapInt32"}:                    struct{}{},
	{"ppc64", "sync/atomic", "CompareAndSwapInt64"}:                    struct{}{},
	{"ppc64", "sync/atomic", "CompareAndSwapUint32"}:                   struct{}{},
	{"ppc64", "sync/atomic", "CompareAndSwapUint64"}:                   struct{}{},
	{"ppc64", "sync/atomic", "CompareAndSwapUintptr"}:                  struct{}{},
	{"ppc64", "sync/atomic", "LoadInt32"}:                              struct{}{},
	{"ppc64", "sync/atomic", "LoadInt64"}:                              struct{}{},
	{"ppc64", "sync/atomic", "LoadPointer"}:                            struct{}{},
	{"ppc64", "sync/atomic", "LoadUint32"}:                             struct{}{},
	{"ppc64", "sync/atomic", "LoadUint64"}:                             struct{}{},
	{"ppc64", "sync/atomic", "LoadUintptr"}:                            struct{}{},
	{"ppc64", "sync/atomic", "StoreInt32"}:                             struct{}{},
	{"ppc64", "sync/atomic", "StoreInt64"}:                             struct{}{},
	{"ppc64", "sync/atomic", "StoreUint32"}:                            struct{}{},
	{"ppc64", "sync/atomic", "StoreUint64"}:                            struct{}{},
	{"ppc64", "sync/atomic", "StoreUintptr"}:                           struct{}{},
	{"ppc64", "sync/atomic", "SwapInt32"}:                              struct{}{},
	{"ppc64", "sync/atomic", "SwapInt64"}:                              struct{}{},
	{"ppc64", "sync/atomic", "SwapUint32"}:                             struct{}{},
	{"ppc64", "sync/atomic", "SwapUint64"}:                             struct{}{},
	{"ppc64", "sync/atomic", "SwapUintptr"}:                            struct{}{},
	{"ppc64le", "internal/runtime/atomic", "And"}:                      struct{}{},
	{"ppc64le", "internal/runtime/atomic", "And8"}:                     struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Cas"}:                      struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Cas64"}:                    struct{}{},
	{"ppc64le", "internal/runtime/atomic", "CasRel"}:                   struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Casint32"}:                 struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Casint64"}:                 struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Casp1"}:                    struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Casuintptr"}:               struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Load"}:                     struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Load64"}:                   struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Load8"}:                    struct{}{},
	{"ppc64le", "internal/runtime/atomic", "LoadAcq"}:                  struct{}{},
	{"ppc64le", "internal/runtime/atomic", "LoadAcq64"}:                struct{}{},
	{"ppc64le", "internal/runtime/atomic", "LoadAcquintptr"}:           struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Loadint32"}:                struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Loadint64"}:                struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Loadp"}:                    struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Loaduint"}:                 struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Loaduintptr"}:              struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Or"}:                       struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Or8"}:                      struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Store"}:                    struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Store64"}:                  struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Store8"}:                   struct{}{},
	{"ppc64le", "internal/runtime/atomic", "StoreRel"}:                 struct{}{},
	{"ppc64le", "internal/runtime/atomic", "StoreRel64"}:               struct{}{},
	{"ppc64le", "internal/runtime/atomic", "StoreReluintptr"}:          struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Storeint32"}:               struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Storeint64"}:               struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Storeuintptr"}:             struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Xadd"}:                     struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Xadd64"}:                   struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Xaddint32"}:                struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Xaddint64"}:                struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Xadduintptr"}:              struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Xchg8"}:                    struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Xchg"}:                     struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Xchg64"}:                   struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Xchgint32"}:                struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Xchgint64"}:                struct{}{},
	{"ppc64le", "internal/runtime/atomic", "Xchguintptr"}:              struct{}{},
	{"ppc64le", "internal/runtime/math", "Add64"}:                      struct{}{},
	{"ppc64le", "internal/runtime/math", "Mul64"}:                      struct{}{},
	{"ppc64le", "internal/runtime/math", "MulUintptr"}:                 struct{}{},
	{"ppc64le", "internal/runtime/sys", "Bswap32"}:                     struct{}{},
	{"ppc64le", "internal/runtime/sys", "Bswap64"}:                     struct{}{},
	{"ppc64le", "internal/runtime/sys", "GetCallerPC"}:                 struct{}{},
	{"ppc64le", "internal/runtime/sys", "GetCallerSP"}:                 struct{}{},
	{"ppc64le", "internal/runtime/sys", "GetClosurePtr"}:               struct{}{},
	{"ppc64le", "internal/runtime/sys", "Len64"}:                       struct{}{},
	{"ppc64le", "internal/runtime/sys", "Len8"}:                        struct{}{},
	{"ppc64le", "internal/runtime/sys", "OnesCount64"}:                 struct{}{},
	{"ppc64le", "internal/runtime/sys", "Prefetch"}:                    struct{}{},
	{"ppc64le", "internal/runtime/sys", "PrefetchStreamed"}:            struct{}{},
	{"ppc64le", "internal/runtime/sys", "TrailingZeros8"}:              struct{}{},
	{"ppc64le", "internal/runtime/sys", "TrailingZeros32"}:             struct{}{},
	{"ppc64le", "internal/runtime/sys", "TrailingZeros64"}:             struct{}{},
	{"ppc64le", "math", "Abs"}:                                         struct{}{},
	{"ppc64le", "math", "Ceil"}:                                        struct{}{},
	{"ppc64le", "math", "Copysign"}:                                    struct{}{},
	{"ppc64le", "math", "FMA"}:                                         struct{}{},
	{"ppc64le", "math", "Floor"}:                                       struct{}{},
	{"ppc64le", "math", "Round"}:                                       struct{}{},
	{"ppc64le", "math", "Trunc"}:                                       struct{}{},
	{"ppc64le", "math", "sqrt"}:                                        struct{}{},
	{"ppc64le", "math/big", "mulWW"}:                                   struct{}{},
	{"ppc64le", "math/bits", "Add"}:                                    struct{}{},
	{"ppc64le", "math/bits", "Add64"}:                                  struct{}{},
	{"ppc64le", "math/bits", "Len"}:                                    struct{}{},
	{"ppc64le", "math/bits", "Len16"}:                                  struct{}{},
	{"ppc64le", "math/bits", "Len32"}:                                  struct{}{},
	{"ppc64le", "math/bits", "Len64"}:                                  struct{}{},
	{"ppc64le", "math/bits", "Len8"}:                                   struct{}{},
	{"ppc64le", "math/bits", "Mul"}:                                    struct{}{},
	{"ppc64le", "math/bits", "Mul64"}:                                  struct{}{},
	{"ppc64le", "math/bits", "OnesCount"}:                              struct{}{},
	{"ppc64le", "math/bits", "OnesCount16"}:                            struct{}{},
	{"ppc64le", "math/bits", "OnesCount32"}:                            struct{}{},
	{"ppc64le", "math/bits", "OnesCount64"}:                            struct{}{},
	{"ppc64le", "math/bits", "OnesCount8"}:                             struct{}{},
	{"ppc64le", "math/bits", "ReverseBytes16"}:                         struct{}{},
	{"ppc64le", "math/bits", "ReverseBytes32"}:                         struct{}{},
	{"ppc64le", "math/bits", "ReverseBytes64"}:                         struct{}{},
	{"ppc64le", "math/bits", "RotateLeft"}:                             struct{}{},
	{"ppc64le", "math/bits", "RotateLeft32"}:                           struct{}{},
	{"ppc64le", "math/bits", "RotateLeft64"}:                           struct{}{},
	{"ppc64le", "math/bits", "Sub"}:                                    struct{}{},
	{"ppc64le", "math/bits", "Sub64"}:                                  struct{}{},
	{"ppc64le", "math/bits", "TrailingZeros8"}:                         struct{}{},
	{"ppc64le", "math/bits", "TrailingZeros16"}:                        struct{}{},
	{"ppc64le", "math/bits", "TrailingZeros32"}:                        struct{}{},
	{"ppc64le", "math/bits", "TrailingZeros64"}:                        struct{}{},
	{"ppc64le", "runtime", "KeepAlive"}:                                struct{}{},
	{"ppc64le", "runtime", "publicationBarrier"}:                       struct{}{},
	{"ppc64le", "runtime", "slicebytetostringtmp"}:                     struct{}{},
	{"ppc64le", "sync", "runtime_LoadAcquintptr"}:                      struct{}{},
	{"ppc64le", "sync", "runtime_StoreReluintptr"}:                     struct{}{},
	{"ppc64le", "sync/atomic", "AddInt32"}:                             struct{}{},
	{"ppc64le", "sync/atomic", "AddInt64"}:                             struct{}{},
	{"ppc64le", "sync/atomic", "AddUint32"}:                            struct{}{},
	{"ppc64le", "sync/atomic", "AddUint64"}:                            struct{}{},
	{"ppc64le", "sync/atomic", "AddUintptr"}:                           struct{}{},
	{"ppc64le", "sync/atomic", "CompareAndSwapInt32"}:                  struct{}{},
	{"ppc64le", "sync/atomic", "CompareAndSwapInt64"}:                  struct{}{},
	{"ppc64le", "sync/atomic", "CompareAndSwapUint32"}:                 struct{}{},
	{"ppc64le", "sync/atomic", "CompareAndSwapUint64"}:                 struct{}{},
	{"ppc64le", "sync/atomic", "CompareAndSwapUintptr"}:                struct{}{},
	{"ppc64le", "sync/atomic", "LoadInt32"}:                            struct{}{},
	{"ppc64le", "sync/atomic", "LoadInt64"}:                            struct{}{},
	{"ppc64le", "sync/atomic", "LoadPointer"}:                          struct{}{},
	{"ppc64le", "sync/atomic", "LoadUint32"}:                           struct{}{},
	{"ppc64le", "sync/atomic", "LoadUint64"}:                           struct{}{},
	{"ppc64le", "sync/atomic", "LoadUintptr"}:                          struct{}{},
	{"ppc64le", "sync/atomic", "StoreInt32"}:                           struct{}{},
	{"ppc64le", "sync/atomic", "StoreInt64"}:                           struct{}{},
	{"ppc64le", "sync/atomic", "StoreUint32"}:                          struct{}{},
	{"ppc64le", "sync/atomic", "StoreUint64"}:                          struct{}{},
	{"ppc64le", "sync/atomic", "StoreUintptr"}:                         struct{}{},
	{"ppc64le", "sync/atomic", "SwapInt32"}:                            struct{}{},
	{"ppc64le", "sync/atomic", "SwapInt64"}:                            struct{}{},
	{"ppc64le", "sync/atomic", "SwapUint32"}:                           struct{}{},
	{"ppc64le", "sync/atomic", "SwapUint64"}:                           struct{}{},
	{"ppc64le", "sync/atomic", "SwapUintptr"}:                          struct{}{},
	{"riscv64", "internal/runtime/atomic", "And"}:                      struct{}{},
	{"riscv64", "internal/runtime/atomic", "And8"}:                     struct{}{},
	{"riscv64", "internal/runtime/atomic", "Cas"}:                      struct{}{},
	{"riscv64", "internal/runtime/atomic", "Cas64"}:                    struct{}{},
	{"riscv64", "internal/runtime/atomic", "CasRel"}:                   struct{}{},
	{"riscv64", "internal/runtime/atomic", "Casint32"}:                 struct{}{},
	{"riscv64", "internal/runtime/atomic", "Casint64"}:                 struct{}{},
	{"riscv64", "internal/runtime/atomic", "Casp1"}:                    struct{}{},
	{"riscv64", "internal/runtime/atomic", "Casuintptr"}:               struct{}{},
	{"riscv64", "internal/runtime/atomic", "Load"}:                     struct{}{},
	{"riscv64", "internal/runtime/atomic", "Load64"}:                   struct{}{},
	{"riscv64", "internal/runtime/atomic", "Load8"}:                    struct{}{},
	{"riscv64", "internal/runtime/atomic", "LoadAcq"}:                  struct{}{},
	{"riscv64", "internal/runtime/atomic", "LoadAcq64"}:                struct{}{},
	{"riscv64", "internal/runtime/atomic", "LoadAcquintptr"}:           struct{}{},
	{"riscv64", "internal/runtime/atomic", "Loadint32"}:                struct{}{},
	{"riscv64", "internal/runtime/atomic", "Loadint64"}:                struct{}{},
	{"riscv64", "internal/runtime/atomic", "Loadp"}:                    struct{}{},
	{"riscv64", "internal/runtime/atomic", "Loaduint"}:                 struct{}{},
	{"riscv64", "internal/runtime/atomic", "Loaduintptr"}:              struct{}{},
	{"riscv64", "internal/runtime/atomic", "Or"}:                       struct{}{},
	{"riscv64", "internal/runtime/atomic", "Or8"}:                      struct{}{},
	{"riscv64", "internal/runtime/atomic", "Store"}:                    struct{}{},
	{"riscv64", "internal/runtime/atomic", "Store64"}:                  struct{}{},
	{"riscv64", "internal/runtime/atomic", "Store8"}:                   struct{}{},
	{"riscv64", "internal/runtime/atomic", "StoreRel"}:                 struct{}{},
	{"riscv64", "internal/runtime/atomic", "StoreRel64"}:               struct{}{},
	{"riscv64", "internal/runtime/atomic", "StoreReluintptr"}:          struct{}{},
	{"riscv64", "internal/runtime/atomic", "Storeint32"}:               struct{}{},
	{"riscv64", "internal/runtime/atomic", "Storeint64"}:               struct{}{},
	{"riscv64", "internal/runtime/atomic", "StorepNoWB"}:               struct{}{},
	{"riscv64", "internal/runtime/atomic", "Storeuintptr"}:             struct{}{},
	{"riscv64", "internal/runtime/atomic", "Xadd"}:                     struct{}{},
	{"riscv64", "internal/runtime/atomic", "Xadd64"}:                   struct{}{},
	{"riscv64", "internal/runtime/atomic", "Xaddint32"}:                struct{}{},
	{"riscv64", "internal/runtime/atomic", "Xaddint64"}:                struct{}{},
	{"riscv64", "internal/runtime/atomic", "Xadduintptr"}:              struct{}{},
	{"riscv64", "internal/runtime/atomic", "Xchg"}:                     struct{}{},
	{"riscv64", "internal/runtime/atomic", "Xchg64"}:                   struct{}{},
	{"riscv64", "internal/runtime/atomic", "Xchgint32"}:                struct{}{},
	{"riscv64", "internal/runtime/atomic", "Xchgint64"}:                struct{}{},
	{"riscv64", "internal/runtime/atomic", "Xchguintptr"}:              struct{}{},
	{"riscv64", "internal/runtime/math", "Add64"}:                      struct{}{},
	{"riscv64", "internal/runtime/math", "Mul64"}:                      struct{}{},
	{"riscv64", "internal/runtime/math", "MulUintptr"}:                 struct{}{},
	{"riscv64", "internal/runtime/sys", "Bswap32"}:                     struct{}{},
	{"riscv64", "internal/runtime/sys", "Bswap64"}:                     struct{}{},
	{"riscv64", "internal/runtime/sys", "GetCallerPC"}:                 struct{}{},
	{"riscv64", "internal/runtime/sys", "GetCallerSP"}:                 struct{}{},
	{"riscv64", "internal/runtime/sys", "GetClosurePtr"}:               struct{}{},
	{"riscv64", "internal/runtime/sys", "Len64"}:                       struct{}{},
	{"riscv64", "internal/runtime/sys", "Len8"}:                        struct{}{},
	{"riscv64", "internal/runtime/sys", "OnesCount64"}:                 struct{}{},
	{"riscv64", "internal/runtime/sys", "TrailingZeros32"}:             struct{}{},
	{"riscv64", "internal/runtime/sys", "TrailingZeros64"}:             struct{}{},
	{"riscv64", "internal/runtime/sys", "TrailingZeros8"}:              struct{}{},
	{"riscv64", "math", "Abs"}:                                         struct{}{},
	{"riscv64", "math", "Copysign"}:                                    struct{}{},
	{"riscv64", "math", "FMA"}:                                         struct{}{},
	{"riscv64", "math", "sqrt"}:                                        struct{}{},
	{"riscv64", "math/big", "mulWW"}:                                   struct{}{},
	{"riscv64", "math/bits", "Add"}:                                    struct{}{},
	{"riscv64", "math/bits", "Add64"}:                                  struct{}{},
	{"riscv64", "math/bits", "Len"}:                                    struct{}{},
	{"riscv64", "math/bits", "Len16"}:                                  struct{}{},
	{"riscv64", "math/bits", "Len32"}:                                  struct{}{},
	{"riscv64", "math/bits", "Len64"}:                                  struct{}{},
	{"riscv64", "math/bits", "Len8"}:                                   struct{}{},
	{"riscv64", "math/bits", "Mul"}:                                    struct{}{},
	{"riscv64", "math/bits", "Mul64"}:                                  struct{}{},
	{"riscv64", "math/bits", "OnesCount"}:                              struct{}{},
	{"riscv64", "math/bits", "OnesCount16"}:                            struct{}{},
	{"riscv64", "math/bits", "OnesCount32"}:                            struct{}{},
	{"riscv64", "math/bits", "OnesCount64"}:                            struct{}{},
	{"riscv64", "math/bits", "OnesCount8"}:                             struct{}{},
	{"riscv64", "math/bits", "ReverseBytes16"}:                         struct{}{},
	{"riscv64", "math/bits", "ReverseBytes32"}:                         struct{}{},
	{"riscv64", "math/bits", "ReverseBytes64"}:                         struct{}{},
	{"riscv64", "math/bits", "RotateLeft"}:                             struct{}{},
	{"riscv64", "math/bits", "RotateLeft16"}:                           struct{}{},
	{"riscv64", "math/bits", "RotateLeft32"}:                           struct{}{},
	{"riscv64", "math/bits", "RotateLeft64"}:                           struct{}{},
	{"riscv64", "math/bits", "RotateLeft8"}:                            struct{}{},
	{"riscv64", "math/bits", "Sub"}:                                    struct{}{},
	{"riscv64", "math/bits", "Sub64"}:                                  struct{}{},
	{"riscv64", "math/bits", "TrailingZeros16"}:                        struct{}{},
	{"riscv64", "math/bits", "TrailingZeros32"}:                        struct{}{},
	{"riscv64", "math/bits", "TrailingZeros64"}:                        struct{}{},
	{"riscv64", "math/bits", "TrailingZeros8"}:                         struct{}{},
	{"riscv64", "runtime", "KeepAlive"}:                                struct{}{},
	{"riscv64", "runtime", "publicationBarrier"}:                       struct{}{},
	{"riscv64", "runtime", "slicebytetostringtmp"}:                     struct{}{},
	{"riscv64", "sync", "runtime_LoadAcquintptr"}:                      struct{}{},
	{"riscv64", "sync", "runtime_StoreReluintptr"}:                     struct{}{},
	{"riscv64", "sync/atomic", "AddInt32"}:                             struct{}{},
	{"riscv64", "sync/atomic", "AddInt64"}:                             struct{}{},
	{"riscv64", "sync/atomic", "AddUint32"}:                            struct{}{},
	{"riscv64", "sync/atomic", "AddUint64"}:                            struct{}{},
	{"riscv64", "sync/atomic", "AddUintptr"}:                           struct{}{},
	{"riscv64", "sync/atomic", "CompareAndSwapInt32"}:                  struct{}{},
	{"riscv64", "sync/atomic", "CompareAndSwapInt64"}:                  struct{}{},
	{"riscv64", "sync/atomic", "CompareAndSwapUint32"}:                 struct{}{},
	{"riscv64", "sync/atomic", "CompareAndSwapUint64"}:                 struct{}{},
	{"riscv64", "sync/atomic", "CompareAndSwapUintptr"}:                struct{}{},
	{"riscv64", "sync/atomic", "LoadInt32"}:                            struct{}{},
	{"riscv64", "sync/atomic", "LoadInt64"}:                            struct{}{},
	{"riscv64", "sync/atomic", "LoadPointer"}:                          struct{}{},
	{"riscv64", "sync/atomic", "LoadUint32"}:                           struct{}{},
	{"riscv64", "sync/atomic", "LoadUint64"}:                           struct{}{},
	{"riscv64", "sync/atomic", "LoadUintptr"}:                          struct{}{},
	{"riscv64", "sync/atomic", "StoreInt32"}:                           struct{}{},
	{"riscv64", "sync/atomic", "StoreInt64"}:                           struct{}{},
	{"riscv64", "sync/atomic", "StoreUint32"}:                          struct{}{},
	{"riscv64", "sync/atomic", "StoreUint64"}:                          struct{}{},
	{"riscv64", "sync/atomic", "StoreUintptr"}:                         struct{}{},
	{"riscv64", "sync/atomic", "SwapInt32"}:                            struct{}{},
	{"riscv64", "sync/atomic", "SwapInt64"}:                            struct{}{},
	{"riscv64", "sync/atomic", "SwapUint32"}:                           struct{}{},
	{"riscv64", "sync/atomic", "SwapUint64"}:                           struct{}{},
	{"riscv64", "sync/atomic", "SwapUintptr"}:                          struct{}{},
	{"s390x", "internal/runtime/atomic", "And"}:                        struct{}{},
	{"s390x", "internal/runtime/atomic", "And8"}:                       struct{}{},
	{"s390x", "internal/runtime/atomic", "Cas"}:                        struct{}{},
	{"s390x", "internal/runtime/atomic", "Cas64"}:                      struct{}{},
	{"s390x", "internal/runtime/atomic", "CasRel"}:                     struct{}{},
	{"s390x", "internal/runtime/atomic", "Casint32"}:                   struct{}{},
	{"s390x", "internal/runtime/atomic", "Casint64"}:                   struct{}{},
	{"s390x", "internal/runtime/atomic", "Casp1"}:                      struct{}{},
	{"s390x", "internal/runtime/atomic", "Casuintptr"}:                 struct{}{},
	{"s390x", "internal/runtime/atomic", "Load"}:                       struct{}{},
	{"s390x", "internal/runtime/atomic", "Load64"}:                     struct{}{},
	{"s390x", "internal/runtime/atomic", "Load8"}:                      struct{}{},
	{"s390x", "internal/runtime/atomic", "LoadAcq"}:                    struct{}{},
	{"s390x", "internal/runtime/atomic", "LoadAcq64"}:                  struct{}{},
	{"s390x", "internal/runtime/atomic", "LoadAcquintptr"}:             struct{}{},
	{"s390x", "internal/runtime/atomic", "Loadint32"}:                  struct{}{},
	{"s390x", "internal/runtime/atomic", "Loadint64"}:                  struct{}{},
	{"s390x", "internal/runtime/atomic", "Loadp"}:                      struct{}{},
	{"s390x", "internal/runtime/atomic", "Loaduint"}:                   struct{}{},
	{"s390x", "internal/runtime/atomic", "Loaduintptr"}:                struct{}{},
	{"s390x", "internal/runtime/atomic", "Or"}:                         struct{}{},
	{"s390x", "internal/runtime/atomic", "Or8"}:                        struct{}{},
	{"s390x", "internal/runtime/atomic", "Store"}:                      struct{}{},
	{"s390x", "internal/runtime/atomic", "Store64"}:                    struct{}{},
	{"s390x", "internal/runtime/atomic", "Store8"}:                     struct{}{},
	{"s390x", "internal/runtime/atomic", "StoreRel"}:                   struct{}{},
	{"s390x", "internal/runtime/atomic", "StoreRel64"}:                 struct{}{},
	{"s390x", "internal/runtime/atomic", "StoreReluintptr"}:            struct{}{},
	{"s390x", "internal/runtime/atomic", "Storeint32"}:                 struct{}{},
	{"s390x", "internal/runtime/atomic", "Storeint64"}:                 struct{}{},
	{"s390x", "internal/runtime/atomic", "StorepNoWB"}:                 struct{}{},
	{"s390x", "internal/runtime/atomic", "Storeuintptr"}:               struct{}{},
	{"s390x", "internal/runtime/atomic", "Xadd"}:                       struct{}{},
	{"s390x", "internal/runtime/atomic", "Xadd64"}:                     struct{}{},
	{"s390x", "internal/runtime/atomic", "Xaddint32"}:                  struct{}{},
	{"s390x", "internal/runtime/atomic", "Xaddint64"}:                  struct{}{},
	{"s390x", "internal/runtime/atomic", "Xadduintptr"}:                struct{}{},
	{"s390x", "internal/runtime/atomic", "Xchg"}:                       struct{}{},
	{"s390x", "internal/runtime/atomic", "Xchg64"}:                     struct{}{},
	{"s390x", "internal/runtime/atomic", "Xchgint32"}:                  struct{}{},
	{"s390x", "internal/runtime/atomic", "Xchgint64"}:                  struct{}{},
	{"s390x", "internal/runtime/atomic", "Xchguintptr"}:                struct{}{},
	{"s390x", "internal/runtime/math", "Add64"}:                        struct{}{},
	{"s390x", "internal/runtime/math", "Mul64"}:                        struct{}{},
	{"s390x", "internal/runtime/sys", "Bswap32"}:                       struct{}{},
	{"s390x", "internal/runtime/sys", "Bswap64"}:                       struct{}{},
	{"s390x", "internal/runtime/sys", "GetCallerPC"}:                   struct{}{},
	{"s390x", "internal/runtime/sys", "GetCallerSP"}:                   struct{}{},
	{"s390x", "internal/runtime/sys", "GetClosurePtr"}:                 struct{}{},
	{"s390x", "internal/runtime/sys", "Len64"}:                         struct{}{},
	{"s390x", "internal/runtime/sys", "Len8"}:                          struct{}{},
	{"s390x", "internal/runtime/sys", "OnesCount64"}:                   struct{}{},
	{"s390x", "internal/runtime/sys", "TrailingZeros32"}:               struct{}{},
	{"s390x", "internal/runtime/sys", "TrailingZeros64"}:               struct{}{},
	{"s390x", "internal/runtime/sys", "TrailingZeros8"}:                struct{}{},
	{"s390x", "math", "Ceil"}:                                          struct{}{},
	{"s390x", "math", "FMA"}:                                           struct{}{},
	{"s390x", "math", "Floor"}:                                         struct{}{},
	{"s390x", "math", "Round"}:                                         struct{}{},
	{"s390x", "math", "RoundToEven"}:                                   struct{}{},
	{"s390x", "math", "Trunc"}:                                         struct{}{},
	{"s390x", "math", "sqrt"}:                                          struct{}{},
	{"s390x", "math/big", "mulWW"}:                                     struct{}{},
	{"s390x", "math/bits", "Add"}:                                      struct{}{},
	{"s390x", "math/bits", "Add64"}:                                    struct{}{},
	{"s390x", "math/bits", "Len"}:                                      struct{}{},
	{"s390x", "math/bits", "Len16"}:                                    struct{}{},
	{"s390x", "math/bits", "Len32"}:                                    struct{}{},
	{"s390x", "math/bits", "Len64"}:                                    struct{}{},
	{"s390x", "math/bits", "Len8"}:                                     struct{}{},
	{"s390x", "math/bits", "Mul"}:                                      struct{}{},
	{"s390x", "math/bits", "Mul64"}:                                    struct{}{},
	{"s390x", "math/bits", "OnesCount"}:                                struct{}{},
	{"s390x", "math/bits", "OnesCount16"}:                              struct{}{},
	{"s390x", "math/bits", "OnesCount32"}:                              struct{}{},
	{"s390x", "math/bits", "OnesCount64"}:                              struct{}{},
	{"s390x", "math/bits", "OnesCount8"}:                               struct{}{},
	{"s390x", "math/bits", "ReverseBytes32"}:                           struct{}{},
	{"s390x", "math/bits", "ReverseBytes64"}:                           struct{}{},
	{"s390x", "math/bits", "RotateLeft"}:                               struct{}{},
	{"s390x", "math/bits", "RotateLeft32"}:                             struct{}{},
	{"s390x", "math/bits", "RotateLeft64"}:                             struct{}{},
	{"s390x", "math/bits", "Sub"}:                                      struct{}{},
	{"s390x", "math/bits", "Sub64"}:                                    struct{}{},
	{"s390x", "math/bits", "TrailingZeros16"}:                          struct{}{},
	{"s390x", "math/bits", "TrailingZeros32"}:                          struct{}{},
	{"s390x", "math/bits", "TrailingZeros64"}:                          struct{}{},
	{"s390x", "math/bits", "TrailingZeros8"}:                           struct{}{},
	{"s390x", "runtime", "KeepAlive"}:                                  struct{}{},
	{"s390x", "runtime", "slicebytetostringtmp"}:                       struct{}{},
	{"s390x", "sync", "runtime_LoadAcquintptr"}:                        struct{}{},
	{"s390x", "sync", "runtime_StoreReluintptr"}:                       struct{}{},
	{"s390x", "sync/atomic", "AddInt32"}:                               struct{}{},
	{"s390x", "sync/atomic", "AddInt64"}:                               struct{}{},
	{"s390x", "sync/atomic", "AddUint32"}:                              struct{}{},
	{"s390x", "sync/atomic", "AddUint64"}:                              struct{}{},
	{"s390x", "sync/atomic", "AddUintptr"}:                             struct{}{},
	{"s390x", "sync/atomic", "CompareAndSwapInt32"}:                    struct{}{},
	{"s390x", "sync/atomic", "CompareAndSwapInt64"}:                    struct{}{},
	{"s390x", "sync/atomic", "CompareAndSwapUint32"}:                   struct{}{},
	{"s390x", "sync/atomic", "CompareAndSwapUint64"}:                   struct{}{},
	{"s390x", "sync/atomic", "CompareAndSwapUintptr"}:                  struct{}{},
	{"s390x", "sync/atomic", "LoadInt32"}:                              struct{}{},
	{"s390x", "sync/atomic", "LoadInt64"}:                              struct{}{},
	{"s390x", "sync/atomic", "LoadPointer"}:                            struct{}{},
	{"s390x", "sync/atomic", "LoadUint32"}:                             struct{}{},
	{"s390x", "sync/atomic", "LoadUint64"}:                             struct{}{},
	{"s390x", "sync/atomic", "LoadUintptr"}:                            struct{}{},
	{"s390x", "sync/atomic", "StoreInt32"}:                             struct{}{},
	{"s390x", "sync/atomic", "StoreInt64"}:                             struct{}{},
	{"s390x", "sync/atomic", "StoreUint32"}:                            struct{}{},
	{"s390x", "sync/atomic", "StoreUint64"}:                            struct{}{},
	{"s390x", "sync/atomic", "StoreUintptr"}:                           struct{}{},
	{"s390x", "sync/atomic", "SwapInt32"}:                              struct{}{},
	{"s390x", "sync/atomic", "SwapInt64"}:                              struct{}{},
	{"s390x", "sync/atomic", "SwapUint32"}:                             struct{}{},
	{"s390x", "sync/atomic", "SwapUint64"}:                             struct{}{},
	{"s390x", "sync/atomic", "SwapUintptr"}:                            struct{}{},
	{"wasm", "internal/runtime/sys", "GetCallerPC"}:                    struct{}{},
	{"wasm", "internal/runtime/sys", "GetCallerSP"}:                    struct{}{},
	{"wasm", "internal/runtime/sys", "GetClosurePtr"}:                  struct{}{},
	{"wasm", "internal/runtime/sys", "Len64"}:                          struct{}{},
	{"wasm", "internal/runtime/sys", "Len8"}:                           struct{}{},
	{"wasm", "internal/runtime/sys", "OnesCount64"}:                    struct{}{},
	{"wasm", "internal/runtime/sys", "TrailingZeros32"}:                struct{}{},
	{"wasm", "internal/runtime/sys", "TrailingZeros64"}:                struct{}{},
	{"wasm", "internal/runtime/sys", "TrailingZeros8"}:                 struct{}{},
	{"wasm", "math", "Abs"}:                                            struct{}{},
	{"wasm", "math", "Ceil"}:                                           struct{}{},
	{"wasm", "math", "Copysign"}:                                       struct{}{},
	{"wasm", "math", "Floor"}:                                          struct{}{},
	{"wasm", "math", "RoundToEven"}:                                    struct{}{},
	{"wasm", "math", "Trunc"}:                                          struct{}{},
	{"wasm", "math", "sqrt"}:                                           struct{}{},
	{"wasm", "math/bits", "Len"}:                                       struct{}{},
	{"wasm", "math/bits", "Len16"}:                                     struct{}{},
	{"wasm", "math/bits", "Len32"}:                                     struct{}{},
	{"wasm", "math/bits", "Len64"}:                                     struct{}{},
	{"wasm", "math/bits", "Len8"}:                                      struct{}{},
	{"wasm", "math/bits", "OnesCount"}:                                 struct{}{},
	{"wasm", "math/bits", "OnesCount16"}:                               struct{}{},
	{"wasm", "math/bits", "OnesCount32"}:                               struct{}{},
	{"wasm", "math/bits", "OnesCount64"}:                               struct{}{},
	{"wasm", "math/bits", "OnesCount8"}:                                struct{}{},
	{"wasm", "math/bits", "RotateLeft"}:                                struct{}{},
	{"wasm", "math/bits", "RotateLeft32"}:                              struct{}{},
	{"wasm", "math/bits", "RotateLeft64"}:                              struct{}{},
	{"wasm", "math/bits", "TrailingZeros16"}:                           struct{}{},
	{"wasm", "math/bits", "TrailingZeros32"}:                           struct{}{},
	{"wasm", "math/bits", "TrailingZeros64"}:                           struct{}{},
	{"wasm", "math/bits", "TrailingZeros8"}:                            struct{}{},
	{"wasm", "runtime", "KeepAlive"}:                                   struct{}{},
	{"wasm", "runtime", "slicebytetostringtmp"}:                        struct{}{},
}

func TestIntrinsics(t *testing.T) {
	cfg := &intrinsicBuildConfig{
		goppc64:   10,
		goriscv64: 23,
	}
	initIntrinsics(cfg)

	if *updateIntrinsics {
		var updatedIntrinsics []*testIntrinsicKey
		for ik, _ := range intrinsics {
			updatedIntrinsics = append(updatedIntrinsics, &testIntrinsicKey{ik.arch.Name, ik.pkg, ik.fn})
		}
		slices.SortFunc(updatedIntrinsics, func(a, b *testIntrinsicKey) int {
			if n := strings.Compare(a.archName, b.archName); n != 0 {
				return n
			}
			if n := strings.Compare(a.pkg, b.pkg); n != 0 {
				return n
			}
			return strings.Compare(a.fn, b.fn)
		})
		for _, tik := range updatedIntrinsics {
			fmt.Printf("\t{%q, %q, %q}: struct{}{},\n", tik.archName, tik.pkg, tik.fn)
		}
		return
	}

	gotIntrinsics := make(map[testIntrinsicKey]struct{})
	for ik, _ := range intrinsics {
		gotIntrinsics[testIntrinsicKey{ik.arch.Name, ik.pkg, ik.fn}] = struct{}{}
	}
	for ik, _ := range gotIntrinsics {
		if _, found := wantIntrinsics[ik]; !found {
			t.Errorf("Got unwanted intrinsic %v %v.%v", ik.archName, ik.pkg, ik.fn)
		}
	}

	for ik, _ := range wantIntrinsics {
		if _, found := gotIntrinsics[ik]; !found {
			t.Errorf("Want missing intrinsic %v %v.%v", ik.archName, ik.pkg, ik.fn)
		}
	}
}

func TestIntrinsicBuilders(t *testing.T) {
	cfg := &intrinsicBuildConfig{}
	initIntrinsics(cfg)

	for _, arch := range sys.Archs {
		if intrinsics.lookup(arch, "internal/runtime/sys", "GetCallerSP") == nil {
			t.Errorf("No intrinsic for internal/runtime/sys.GetCallerSP on arch %v", arch)
		}
	}

	if intrinsics.lookup(sys.ArchAMD64, "runtime", "slicebytetostringtmp") == nil {
		t.Error("No intrinsic for runtime.slicebytetostringtmp")
	}

	if intrinsics.lookup(sys.ArchRISCV64, "runtime", "publicationBarrier") == nil {
		t.Errorf("No intrinsic for runtime.publicationBarrier on arch %v", sys.ArchRISCV64)
	}

	if intrinsics.lookup(sys.ArchAMD64, "internal/runtime/sys", "Bswap32") == nil {
		t.Errorf("No intrinsic for internal/runtime/sys.Bswap32 on arch %v", sys.ArchAMD64)
	}
	if intrinsics.lookup(sys.ArchAMD64, "internal/runtime/sys", "Bswap64") == nil {
		t.Errorf("No intrinsic for internal/runtime/sys.Bswap64 on arch %v", sys.ArchAMD64)
	}

	if intrinsics.lookup(sys.ArchPPC64, "internal/runtime/sys", "Bswap64") != nil {
		t.Errorf("Found intrinsic for internal/runtime/sys.Bswap64 on arch %v", sys.ArchPPC64)
	}

	cfg.goppc64 = 10
	cfg.instrumenting = true

	initIntrinsics(cfg)

	if intrinsics.lookup(sys.ArchAMD64, "runtime", "slicebytetostringtmp") != nil {
		t.Error("Intrinsic incorrectly exists for runtime.slicebytetostringtmp")
	}

	if intrinsics.lookup(sys.ArchPPC64, "internal/runtime/sys", "Bswap64") == nil {
		t.Errorf("No intrinsic for internal/runtime/sys.Bswap64 on arch %v", sys.ArchPPC64)
	}
}
