"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.postProcessing = postProcessing;
exports.registerData2SummaryRoutes = registerData2SummaryRoutes;
exports.registerSummaryAssistantRoutes = registerSummaryAssistantRoutes;
var _configSchema = require("@osd/config-schema");
var _llm = require("../../common/constants/llm");
var _get_opensearch_client_transport = require("../utils/get_opensearch_client_transport");
var _get_agent = require("./get_agent");
var _index_type_detect = require("./index_type_detect");
var _error_handler = require("./error_handler");
var _errors = require("./errors");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const SUMMARY_AGENT_CONFIG_ID = 'os_summary';
const LOG_PATTERN_SUMMARY_AGENT_CONFIG_ID = 'os_summary_with_log_pattern';
const DATA2SUMMARY_AGENT_CONFIG_ID = 'os_data2summary';
function postProcessing(output) {
  const pattern = /<summarization>(.*?)<\/summarization>.*?<final insights>(.*?)<\/final insights>/s;
  const match = output.match(pattern);
  if (match) {
    const [, summarization, finalInsights] = match;
    const processedOutput = `${summarization.trim()}\n${finalInsights.trim()}`;
    return processedOutput;
  }
  return output;
}
function registerSummaryAssistantRoutes(router, assistantService) {
  router.post({
    path: _llm.SUMMARY_ASSISTANT_API.SUMMARIZE,
    validate: {
      body: _configSchema.schema.object({
        summaryType: _configSchema.schema.string(),
        insightType: _configSchema.schema.maybe(_configSchema.schema.string()),
        question: _configSchema.schema.string(),
        context: _configSchema.schema.maybe(_configSchema.schema.string()),
        index: _configSchema.schema.maybe(_configSchema.schema.string()),
        dsl: _configSchema.schema.maybe(_configSchema.schema.string()),
        topNLogPatternData: _configSchema.schema.maybe(_configSchema.schema.string())
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, router.handleLegacyErrors(async (context, req, res) => {
    try {
      var _response$body$infere;
      const client = await (0, _get_opensearch_client_transport.getOpenSearchClientTransport)({
        context,
        dataSourceId: req.query.dataSourceId
      });
      const assistantClient = assistantService.getScopedClient(req, context);
      let isLogIndex = false;
      if (req.body.index) {
        isLogIndex = await (0, _index_type_detect.detectIndexType)(client, assistantClient, req.body.index, req.query.dataSourceId);
      }
      const agentConfigId = req.body.index && req.body.dsl && isLogIndex ? LOG_PATTERN_SUMMARY_AGENT_CONFIG_ID : SUMMARY_AGENT_CONFIG_ID;
      const response = await assistantClient.executeAgentByConfigName(agentConfigId, {
        context: req.body.context,
        question: req.body.question,
        index: req.body.index,
        input: req.body.dsl,
        topNLogPatternData: req.body.topNLogPatternData
      });
      let insightAgentIdExists = false;
      if (req.body.insightType) {
        insightAgentIdExists = !!(await detectInsightAgentId(req.body.insightType, req.body.summaryType, client));
      }
      const summary = (_response$body$infere = response.body.inference_results[0]) === null || _response$body$infere === void 0 || (_response$body$infere = _response$body$infere.output[0]) === null || _response$body$infere === void 0 ? void 0 : _response$body$infere.result;
      if (!summary) {
        return res.customError({
          body: 'Execute agent failed with empty response!',
          statusCode: 500
        });
      }
      return res.ok({
        body: {
          summary,
          insightAgentIdExists
        }
      });
    } catch (e) {
      return (0, _error_handler.handleError)(e, res, context.assistant_plugin.logger);
    }
  }));
  router.post({
    path: _llm.SUMMARY_ASSISTANT_API.INSIGHT,
    validate: {
      body: _configSchema.schema.object({
        summaryType: _configSchema.schema.string(),
        insightType: _configSchema.schema.string(),
        summary: _configSchema.schema.string(),
        context: _configSchema.schema.string(),
        question: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, router.handleLegacyErrors(async (context, req, res) => {
    try {
      var _response$body$infere2;
      const client = await (0, _get_opensearch_client_transport.getOpenSearchClientTransport)({
        context,
        dataSourceId: req.query.dataSourceId
      });
      const assistantClient = assistantService.getScopedClient(req, context);
      const insightAgentId = await detectInsightAgentId(req.body.insightType, req.body.summaryType, client);
      if (!insightAgentId) {
        return res.notFound({
          body: 'Agent not found'
        });
      }
      const response = await assistantClient.executeAgent(insightAgentId, {
        context: req.body.context,
        summary: req.body.summary,
        question: req.body.question
      });
      const insight = (_response$body$infere2 = response.body.inference_results[0]) === null || _response$body$infere2 === void 0 || (_response$body$infere2 = _response$body$infere2.output[0]) === null || _response$body$infere2 === void 0 ? void 0 : _response$body$infere2.result;
      if (!insight) {
        return res.customError({
          body: 'Execute agent failed with empty response!',
          statusCode: 500
        });
      }
      return res.ok({
        body: insight
      });
    } catch (e) {
      return (0, _error_handler.handleError)(e, res, context.assistant_plugin.logger);
    }
  }));
}
async function detectInsightAgentId(insightType, summaryType, client) {
  // for user_insight, we can
  // only get it by searching on name since it is not stored in agent config.
  try {
    if (insightType === 'user_insight' && summaryType === 'alerts') {
      return await (0, _get_agent.searchAgent)({
        name: 'KB_For_Alert_Insight'
      }, client);
    }
  } catch (e) {
    // It only detects if the agent exists, we don't want to throw the error when not found the agent
    // So we return `undefined` to indicate the insight agent id not found
    if (e instanceof _errors.AgentNotFoundError) {
      return undefined;
    }
    throw e;
  }
}
function registerData2SummaryRoutes(router, assistantService) {
  router.post({
    path: _llm.SUMMARY_ASSISTANT_API.DATA2SUMMARY,
    validate: {
      body: _configSchema.schema.object({
        sample_data: _configSchema.schema.string(),
        sample_count: _configSchema.schema.maybe(_configSchema.schema.number()),
        total_count: _configSchema.schema.maybe(_configSchema.schema.number()),
        question: _configSchema.schema.maybe(_configSchema.schema.string()),
        ppl: _configSchema.schema.maybe(_configSchema.schema.string())
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, router.handleLegacyErrors(async (context, req, res) => {
    const assistantClient = assistantService.getScopedClient(req, context);
    try {
      const response = await assistantClient.executeAgentByConfigName(DATA2SUMMARY_AGENT_CONFIG_ID, {
        sample_data: req.body.sample_data,
        total_count: req.body.total_count,
        sample_count: req.body.sample_count,
        ppl: req.body.ppl,
        question: req.body.question
      });
      let result = response.body.inference_results[0].output[0].result;
      result = postProcessing(result);
      if (result) {
        return res.ok({
          body: result
        });
      } else {
        return res.customError({
          body: 'Execute agent failed with empty response!',
          statusCode: 500
        });
      }
    } catch (e) {
      return (0, _error_handler.handleError)(e, res, context.assistant_plugin.logger);
    }
  }));
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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